/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <memory>
#include <utility>
#include <vector>
#include <agxSDK/Assembly.h>
#include <openplx/Object.h>
#include <openplx/OpenPlxCoreApi.h>
#include <agxOpenPLX/AgxCache.h>
#include <agxOpenPLX/OpenPlxToAgxMapper.h>
#include <agxOpenPLX/SignalQueue.h>
#include <agxOpenPLX/OptParams.h>
#include <agxOpenPLX/export.h>

namespace agxopenplx
{

  class OpenPlxToAgxVisualsMapper;

  /**
   * \brief The return type of functions that load a .openplx file. Bundles
   * all the resulting data of loading a .openplx scene.
   */
  class AGXOPENPLX_EXPORT LoadResult
  {
    public:
      LoadResult() = default;

      LoadResult(
        agxSDK::Assembly* assembly, std::shared_ptr<openplx::Core::Api::OpenPlxContext> context,
        std::shared_ptr<openplx::Core::Object> scene, openplx::Errors errors,
        std::shared_ptr<agxopenplx::OpenPlxToAgxMapper> to_agx_mapper,
        std::shared_ptr<agxopenplx::OpenPlxToAgxVisualsMapper> to_agx_visuals_mapper, bool visuals_added);

      /**
       * \return The assembly into which the openplx scene has been loaded
       */
      agxSDK::Assembly* assembly();

      /**
       * \return The power line into which the openplx drive train has been loaded
       */
      agxPowerLine::PowerLine* power_line();

      /**
       * \return The openplx root instance (usually a Physics3D.System)
       */
      std::shared_ptr<openplx::Core::Object> scene();

      /**
       * \return The openplx errors that was found when parsing/loading the .openplx file
       */
      openplx::Errors errors();

      /**
       * \return The openplx core context
       */
      std::shared_ptr<openplx::Core::Api::OpenPlxContext> context();

      /**
       *\return The agxOpenPLX related runtime metadata.
       */
      std::shared_ptr<agxopenplx::AgxMetadata> metadata();

      /**
       * \return The input signal queue if signalling is configured, null otherwise
       */
      std::shared_ptr<agxopenplx::InputSignalQueue> getInputSignalQueue();

      /**
       * \return The output signal queue if signalling is configured, null otherwise
       */
      std::shared_ptr<agxopenplx::OutputSignalQueue> getOutputSignalQueue();

      /**
       * \brief True if the loaded model contained any visuals
       */
      bool visualsAdded();

      DOXYGEN_START_INTERNAL_BLOCK()
      /**
       * \return The OpenPlxToAgxMapper instance (used internally)
       */
      std::shared_ptr<agxopenplx::OpenPlxToAgxMapper> toAgxMapper();

      /**
       * \return The OpenPlxToAgxVisualsMapper instance (used internally)
       */
      std::shared_ptr<agxopenplx::OpenPlxToAgxVisualsMapper> toAgxVisualsMapper();

      /**
       * \brief Adds a input signal queue to the result (used internally)
       */
      void addInputSignalQueue(std::shared_ptr<agxopenplx::InputSignalQueue> input_queue);

      /**
       * \brief Adds a output signal queue to the result (used internally)
       */
      void addOutputSignalQueue(std::shared_ptr<agxopenplx::OutputSignalQueue> output_queue);
      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agxSDK::AssemblyRef m_assembly;
      std::shared_ptr<openplx::Core::Api::OpenPlxContext> m_context;
      std::shared_ptr<openplx::Core::Object> m_scene;
      openplx::Errors m_errors;
      std::shared_ptr<agxopenplx::OpenPlxToAgxMapper> m_to_agx_mapper;
      std::shared_ptr<agxopenplx::OpenPlxToAgxVisualsMapper> m_to_agx_visuals_mapper;
      std::shared_ptr<agxopenplx::InputSignalQueue> m_input_queue;
      std::shared_ptr<agxopenplx::OutputSignalQueue> m_output_queue;
      bool m_visuals_added;
  };

  /**
   * @brief Loads scene from file and report errors if any
   *
   * Registers bundles, parses OpenPLX and maps OpenPLX to agx
   *
   * @param simulation The agx simulation to load the scene into
   * @param source The .openplx source text
   * @param optionalParameters Optional parameters, see agxopenplx::OptParams
   */
  AGXOPENPLX_EXPORT LoadResult load_from_string(
    agxSDK::Simulation* simulation, const std::string& source, OptParams optionalParameters = OptParams());

  /**
   * @brief Loads scene from string and report errors if any
   * Registers bundles, parses OpenPLX and maps OpenPLX to agx
   *
   * @param simulation The agx simulation to load the scene into
   * @param path The path to the .openplx file
   * @param optionalParameters Optional parameters, see agxopenplx::OptParams
   */
  AGXOPENPLX_EXPORT LoadResult load_from_file(
    agxSDK::Simulation* simulation, const std::string& path, OptParams optionalParameters = OptParams());

  /**
   * Serializes the runtime tree to JSON
   */
  AGXOPENPLX_EXPORT std::string serialize_file(
    const std::string& path, OptParams optionalParameters = OptParams());

  /**
   * Bakes a .openplx file and all of its dependencies into a folder with a single .openplx file
   * and all externel dependencies that can be loaded by itself.
   * @param path The path to the .openplx file
   * @param output_dir The path to the folder where the baked assets will be placed
   * @param bake_imports If true all imports will be baked into .openplx and the import declarations removed
   * @param bundle_paths The bundle paths
   * @return true if the bake was successful
   */
  AGXOPENPLX_EXPORT bool bake_file(const std::string& path, const std::string& output_dir, bool bake_imports, const std::vector<std::string>& bundle_paths);

  /**
   * @brief Register all known OpenPLX bundles
   */
  AGXOPENPLX_EXPORT void register_bundles(openplx::Core::Api::OpenPlxContext& ctx);

  /**
   * @brief Register all known OpenPLX plugins
   */
  AGXOPENPLX_EXPORT void register_plugins(
    openplx::Core::Api::OpenPlxContext& ctx, std::shared_ptr<agxopenplx::AgxCache> cache = nullptr);

  /**
   * @brief Set openplx log level
   *
   * @param log_level_name One of "trace", "debug", "info", "warn", "error", "critical", or "off".
   */
  AGXOPENPLX_EXPORT void set_log_level(const std::string& log_level_name);

  AGXOPENPLX_EXPORT std::pair<std::shared_ptr<openplx::Core::Object>, openplx::Errors> parse_and_evaluate_string(
    const std::string& bundle_paths, const std::string& source, const std::string& modelname);

  AGXOPENPLX_EXPORT std::string parseWithPlugin(std::shared_ptr<openplx::Plugin>, const std::string& key);

  /**
   * Converts errors to a vector of strings
   * @return a vector of error messages
   */
  AGXOPENPLX_EXPORT std::vector<std::string> get_error_strings(const openplx::Errors& errors);
}
