/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <memory>
#include <unordered_map>
#include <agx/ElementaryConstraint.h>
#include <agx/Logger.h>
#include <agx/Referenced.h>
#include <agx/ref_ptr.h>
#include <agxPowerLine/Connector.h>
#include <agxPowerLine/PowerLine.h>
#include <agxSDK/Assembly.h>
#include <agxSensor/Environment.h>
#include <openplx/UuidGenerator.h>
#include <agxOpenPLX/export.h>

namespace openplx::Core
{
  class Object;
}

namespace agxopenplx
{

  enum class AgxObjectMapMode
  {
    Name,
    Uuid
  };

  /**
   * Used to map OpenPLX instances to AGX instances either using the OpenPLX
   * instance UUID remapped with the OpenPLX instance type using UUIDv5,
   * or the OpenPLX instance name.
   */
  class AGXOPENPLX_EXPORT AgxObjectMap
  {
    public:
      /**
       * Creates a signal source mapper that looks up AGX instances from an assembly and a power line.
       * \param assembly - The assembly used to lookup AGX instances from OpenPLX instances
       * \param power_line - The power line used to lookup AGX instances from OpenPLX instances
       * \param map_mode - The method to map OpenPLX instances to AGX instances (by Uuid by default).
       */
      static std::shared_ptr<AgxObjectMap> create(
        agxSDK::Assembly* assembly, agxPowerLine::PowerLine* power_line = nullptr,
        agxSensor::Environment* environment = nullptr, AgxObjectMapMode map_mode = AgxObjectMapMode::Uuid);

      /**
       * Creates a signal source mapper that looks up AGX instances from pre filled map.
       * \param ref_map - The pre filled map mapping keys to AGX instances
       * \param map_mode - The method to map OpenPLX instances to keys for the ref_map (Uuid by default).
       */
      static std::shared_ptr<AgxObjectMap> createPreMapped(
        std::unordered_multimap<std::string, agx::Referenced*> ref_map,
        AgxObjectMapMode map_mode = AgxObjectMapMode::Uuid);

      /**
       * Finds an AGX instances from an OpenPLX instance
       * \param object_or_trait - The OpenPLX instance
       */
      template <class T, class S>
      agx::ref_ptr<T> findMapped(const std::shared_ptr<S>& object_or_trait)
      {
        static_assert(std::is_base_of<agx::Referenced, T>::value, "T must derive from agx::Referenced");
        auto object = std::dynamic_pointer_cast<openplx::Core::Object>(object_or_trait);
        if (object == nullptr)
          return nullptr;
        auto raw_matches = findMappedRaw(object);
        if (!raw_matches.empty()) {
          for (auto& raw : raw_matches) {
            auto specific = dynamic_cast<T*>(raw);
            if (specific != nullptr) {
              return agx::ref_ptr<T>(specific);
            }
          }
        }
        if (m_assembly != nullptr) {
          if constexpr (std::is_same_v<T, agx::ObserverFrame>) {
            if (m_map_mode == AgxObjectMapMode::Uuid && object->getType() != nullptr) {
              auto uuid = agx::Uuid(
                openplx::UuidGenerator::generateUuidv5(
                  object->getUuid(), object->getType()->getNameWithNamespace(".")));
              return m_assembly->getObserverFrame(uuid);
            }
            else if (m_map_mode == AgxObjectMapMode::Name) {
              return m_assembly->getObserverFrame(object->getName());
            }
          }
          else if constexpr (std::is_same_v<T, agx::RigidBody>) {
            if (m_map_mode == AgxObjectMapMode::Uuid && object->getType() != nullptr) {
              auto uuid = agx::Uuid(
                openplx::UuidGenerator::generateUuidv5(
                  object->getUuid(), object->getType()->getNameWithNamespace(".")));
              return m_assembly->getRigidBody(uuid);
            }
            else if (m_map_mode == AgxObjectMapMode::Name) {
              return m_assembly->getRigidBody(object->getName());
            }
          }
          else if constexpr (std::is_base_of_v<agx::Constraint, T>) {
            if (m_map_mode == AgxObjectMapMode::Uuid && object->getType() != nullptr) {
              auto uuid = agx::Uuid(
                openplx::UuidGenerator::generateUuidv5(
                  object->getUuid(), object->getType()->getNameWithNamespace(".")));
              agx::ConstraintRef constraint = m_assembly->getConstraint(uuid);
              return agx::dynamic_pointer_cast<T>(constraint);
            }
            else if (m_map_mode == AgxObjectMapMode::Name) {
              agx::ConstraintRef constraint = m_assembly->getConstraint(object->getName());
              return agx::dynamic_pointer_cast<T>(constraint);
            }
          }
          else if constexpr (std::is_base_of_v<agxSDK::Assembly, T>) {
            if (m_map_mode == AgxObjectMapMode::Uuid && object->getType() != nullptr) {
              auto uuid = agx::Uuid(
                openplx::UuidGenerator::generateUuidv5(
                  object->getUuid(), object->getType()->getNameWithNamespace(".")));
              agxSDK::AssemblyRef assembly = m_assembly->getAssembly(uuid);
              return agx::dynamic_pointer_cast<T>(assembly);
            }
            else if (m_map_mode == AgxObjectMapMode::Name) {
              agxSDK::AssemblyRef assembly = m_assembly->getAssembly(object->getName());
              return agx::dynamic_pointer_cast<T>(assembly);
            }
          }
        }
        LOGGER_WARNING() << "Unable to map " << object->getName() << " to " << typeid(T).name() << LOGGER_ENDL();
        return nullptr;
      }

      DOXYGEN_START_INTERNAL_BLOCK()
      std::vector<agx::Referenced*> findMappedRaw(const std::shared_ptr<openplx::Core::Object>& object);

      static void populateWithConstraints(
        agxSDK::Assembly& assembly, std::unordered_multimap<std::string, agx::Referenced*>& ref_map,
        AgxObjectMapMode map_mode);

      static void populateWithSecondaryConstraints(
        agxSDK::Assembly& assembly, std::unordered_multimap<std::string, agx::Referenced*>& ref_map,
        AgxObjectMapMode map_mode);

      static void populateWithUnits(
        agxPowerLine::PowerLine& power_line, std::unordered_multimap<std::string, agx::Referenced*>& ref_map,
        AgxObjectMapMode map_mode);

      static void populateWithConnectors(
        agxPowerLine::PowerLine& power_line, std::unordered_multimap<std::string, agx::Referenced*>& ref_map,
        AgxObjectMapMode map_mode);

      static void populateWithSensors(
        agxSensor::Environment& environment, std::unordered_multimap<std::string, agx::Referenced*>& ref_map,
        AgxObjectMapMode map_mode);

      static void populateWithShovels(
        agxSDK::Assembly& assembly, std::unordered_multimap<std::string, agx::Referenced*>& ref_map,
        AgxObjectMapMode map_mode);

      static void populateWithFrames(
        agxSDK::Assembly& assembly, std::unordered_multimap<std::string, agx::Referenced*>& ref_map,
        AgxObjectMapMode map_mode);
      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      AgxObjectMap(AgxObjectMapMode map_mode);
      agxSDK::AssemblyRef m_assembly;
      std::unordered_multimap<std::string, agx::Referenced*> m_ref_map;
      AgxObjectMapMode m_map_mode;
  };
}
