/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxDriveTrain/Shaft.h>
#include <agxDriveTrain/VelocityConstraint.h>
#include <agxPowerLine/PowerLine.h>


namespace agxDriveTrain
{

  class AGXMODEL_EXPORT Brake : public agxDriveTrain::Shaft
  {
  public:
    /**
    This model represents a brake system with dry friction between a brake pad
    and a rotational plate. The input shaft is connected to the rotating plate, while
    the brake pad is fixed but can be moved closer to or further from the plate using
    a spring-loaded mechanism. This mechanism applies a normal force that increases
    as the pad moves closer to the plate, creating the concept of braking torque capacity.

    When the brake pad is fully disengaged, no force is applied to the plate,
    resulting in no braking. When the pad is fully engaged, maximum normal force
    is applied to the plate, generating maximum braking torque. 

    The "engagement fraction" here represents the proportion of the applied braking force
    relative to the maximum torque capacity. A value of 1 indicates full engagement
    (maximum braking force), while 0 represents no braking force.

    Similar to the dry clutch, the brake model includes a "locked" mode, where a clamp
    is activated to prevent the plate from rotating. The lock remains engaged
    until the brake is disengaged or if external forces cause the plate to rotate
    in the opposite direction, as with dynamic braking scenarios. This "locked" mode
    functions as a "parking brake".

    This brake model has two operational modes: manual and automatic. In manual mode,
    the user directly controls the engagement fraction or activates the lock mechanism
    using a control algorithm and step event listener. The brake pedal can be engaged or
    disengaged manually by the user.

    In automatic mode, the engagement is handled by a time constant, which dictates the
    transition between fully disengaged (no braking) and fully engaged (maximum braking).
    The function "engage" controls the increase or decrease in braking force. If the auto
    lock mode is active, logic determines when to engage or disengage the lock based on slip
    velocity. If the slip velocity is sufficiently small compared to the input shaft velocity,
    the lock is engaged. The lock is disengaged if either the function "engage(false)" is
    called or if the brake torque becomes negative, indicating dynamic braking.

    Automatic engagement causes the engagement fraction to increase linearly from the moment
    "engage(true)" is called, or decrease linearly from the moment "engage(false)" is called.

    For this model, we assume that the brake pad behaves like a linear spring, so the normal
    force is directly proportional to the pad's position relative to the plate.
    */
    Brake();

    /**
    Puts the brake pedal in assisted mode so it will fully engage or disengage in a
    time period determined by setTimeConstant below. Using setEngage(true) will turn
    manual mode to false as per the rule: respect the users's intention.
    \param engage - true to engage, false to disengage from current fraction (Default: False).
    */
    void setEngage(bool engage);

    /**
    \return - Returns whether the brake is engaged or not.
    */
    bool isEngaged() const;

    /**
    \param fraction - Set the fraction of the brake pedal opening in range [0,1] (Default: 0.0).
    */
    void setFraction(agx::Real fraction);

    /**
    \return - Returns the fraction of the brake pedal opening (fraction of maximum normal
    force betwen the friction plates).
    */
    agx::Real getFraction() const;

    /**
    Set the time constant for the brake engagement. Engagement
    is linear and proceeds by a fraction of +/- step/tau per step. The default
    time constant is in seconds.
    \param tau - the time constant. The unit of tau is the same as the unit of time step size defined by the user (Default: 2.5).
    */
    void setEngageTimeConstant(agx::Real tau);

    /**
    Set the time constant for brake disengagement. Disengagement
    is linear and proceeds by a fraction of +/- step/tau per step. The default
    time constant is in seconds.
    \param tau - the time constant. The unit of tau is the same as the unit of time step size defined by the user (Default: 1.0).
    */
    void setDisengageTimeConstant(agx::Real tau);

    /**
    When manual mode is true, user must control all aspects of the brake
    engagement and disengagement. Using the engage method will turn this flag to false.
    \param manual - Set or unset manual mode (Default: False).
    */
    void setManualMode(bool manual);

    /**
    \return - Returns true in manual mode, false otherwise.
    */
    bool isManualMode() const;

    /**
    \return - Returns the time constant of the brake for engaging.
    */
    agx::Real getEngageTimeConstant() const;

    /**
    \return - Returns the time constant of the brake for disengaging.
    */
    agx::Real getDisengageTimeConstant() const;

    /**
    Lock or unlock the brake. In locked mode, it will behave as a parking brake.
    \param setLock - desired state (Default: False).
    */
    void setLock(bool setLock);

    /**
    Returns the lock status.
    */
    bool isLocked() const;

    /**
    This is the maximum torque that the brake plates can transfer: it will slip if
    this MinRelativeSlip is exceeded. Set the torque capacity directly here (Default: 2000).
    */
    void setTorqueCapacity(agx::Real capacity);

    /**
    \return - Returns the torque capacity of the brake.
    */
    agx::Real getTorqueCapacity() const;

    /**
    In auto lock mode, the friction plates will lock when the slip is small enough.
    \param autoLock - true or false to enable or disable autoLock, respectively (Default: False).
    */
    void setAutoLock(bool autoLock);

    /**
    \return - State of auto lock.
    */
    bool isAutoLock() const;

    /**
    \return - Torque transferred between the friction plates.
    */
    agx::Real getTorque() const;

    /**
    \return - Angular velocity of input shaft.
    */
    agx::Real getInputVelocity() const;

    /**
    In autolock mode, the friction plates will lock with the ratio of slip is less than this threshold.
    \param minRelativeSlip - the threshold of the minimum slip (Default: 1e-5).
    */
    void  setMinRelativeSlip(agx::Real minRelativeSlip);

    /**
    \return - Minimum relative velocity for lock.
    */
    agx::Real getMinRelativeSlip() const;

    /**
    * \return The VelocityConstraint of the brake.
    */
    const agxDriveTrain::VelocityConstraint* getConstraint() const;

    /**
    * \return The VelocityConstraint of the brake.
    */
    agxDriveTrain::VelocityConstraint* getConstraint();

    DOXYGEN_START_INTERNAL_BLOCK()

    bool preUpdate(agx::Real timeStep) override;

    /**
    Add notification.
    */
    virtual bool addNotification(agxSDK::Simulation* simulation) override;

    /**
    Remove notification.
    */
    virtual bool removeNotification(agxUtil::ConstraintHolder* constraintHolder, agxSDK::Simulation* simulation) override;

    /**
    Stores internal data into stream.
    */
    virtual bool store(agxStream::StorageStream& str) const override;

    /**
    Store velocity constraint data to stream.
    */
    virtual bool postStore(agxStream::StorageStream& str) const override;

    /**
    Restores internal data from stream.
    */
    virtual bool restore(agxStream::StorageStream& str) override;

    /**
    Restore velocity constraint data from stream.
    */
    virtual bool postRestore(agxStream::StorageStream& str) override;

    AGXSTREAM_DECLARE_SERIALIZABLE(agxDriveTrain::Brake);
    DOXYGEN_END_INTERNAL_BLOCK()

  protected:
    virtual ~Brake();

    agxDriveTrain::VelocityConstraintRef m_velocityConstraint;
    agx::Real m_torqueCapacity;
    bool      m_isEngaged;
    bool      m_isAutoLock;
    bool      m_isManual;
    agx::Real m_engageTimeConstant;
    agx::Real m_disengageTimeConstant;
    agx::Real m_minRelativeSlip;
    agx::Real m_fraction;
    bool      m_isLocked;
  };
  typedef agx::ref_ptr< Brake > BrakeRef;
}
