/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Logger.h>

#include <agx/Json.h>
#include <agx/String.h>

namespace agx
{

  /**
  A template class that can be used to read/write a json property and set/get the value in another object.
  */
  template<typename T>
  class GenericJsonParameter {
    public:

      using PropertyGetInt = std::function< int( const T* ) >;
      using PropertyGetBool = std::function< bool( const T* ) >;
      using PropertyGetString = std::function< agx::String( const T* ) >;
      using PropertyGetReal = std::function< agx::Real( const T* ) >;
      using PropertyGetVec3 = std::function< agx::Vec3( const T* ) >;

      using PropertySetInt = std::function< void(int, T* ) >;
      using PropertySetBool = std::function< void( bool, T* ) >;
      using PropertySetString = std::function< void( agx::String, T* ) >;
      using PropertySetReal = std::function< void( agx::Real, T* ) >;
      using PropertySetVec3 = std::function< void(agx::Vec3, T*) >;


      /**
      Type of parameter that should be handled.
      */
      enum ParameterType {
        UNDEFINED_TYPE,
        SCALAR_INT,
        SCALAR_BOOL,
        SCALAR_REAL,
        STRING,
        ARRAY_VEC3
      };

      explicit GenericJsonParameter(const agx::String& name, PropertySetInt setter, PropertyGetInt getter)
        : m_type(SCALAR_INT), m_name(name), m_getInt(getter), m_setInt(setter)
      {
      }
      
      explicit GenericJsonParameter(const agx::String& name, PropertySetBool setter, PropertyGetBool getter)
        : m_type(SCALAR_BOOL), m_name(name), m_getBool(getter), m_setBool(setter)
      {
      }
      
      explicit GenericJsonParameter( const agx::String& name, PropertySetReal setter, PropertyGetReal getter )
        : m_type(SCALAR_REAL), m_name(name), m_getReal(getter), m_setReal(setter)
      {
      }

      explicit GenericJsonParameter(const agx::String& name, PropertySetString setter, PropertyGetString getter)
        : m_type(STRING), m_name(name), m_getString(getter), m_setString(setter)
      {
      }

      explicit GenericJsonParameter(const agx::String& name, PropertySetVec3 setter, PropertyGetVec3 getter, unsigned int numValues)
        : m_type(ARRAY_VEC3), m_name(name), m_numValues(numValues), m_getVec3(getter), m_setVec3(setter)
      {
      }
      
      
      explicit GenericJsonParameter(const agx::String& name, const agx::String& group, PropertySetInt setter, PropertyGetInt getter)
        : m_type(SCALAR_INT), m_name(name), m_group(group), m_getInt(getter), m_setInt(setter)
      {
      }

      explicit GenericJsonParameter(const agx::String& name, const agx::String& group, PropertySetBool setter, PropertyGetBool getter)
        : m_type(SCALAR_BOOL), m_name(name), m_group(group), m_getBool(getter), m_setBool(setter)
      {
      }

      explicit GenericJsonParameter( const agx::String& name, const agx::String& group, PropertySetReal setter, PropertyGetReal getter )
        : m_type(SCALAR_REAL), m_name(name), m_group(group), m_getReal(getter), m_setReal(setter)
      {
      }

      explicit GenericJsonParameter(const agx::String& name, const agx::String& group, PropertySetString setter, PropertyGetString getter)
        : m_type(STRING), m_name(name), m_group(group), m_getString(getter), m_setString(setter)
      {
      }

      explicit GenericJsonParameter(const agx::String& name, const agx::String& group, PropertySetVec3 setter, PropertyGetVec3 getter)
        : m_type(ARRAY_VEC3), m_name(name), m_group(group), m_getVec3(getter), m_setVec3(setter)
      {
      }
      
      /**
      Is value marked as "default"?
      */
      bool isDefault( const agxJson::Value& value ) const
      {
        bool def = false;
        if ( value.isString() ) {
          def = agx::String(value.asString()).lower() == "default";
        }
        return def;
      }

      /**
      Read an int from a json value.
      */
      int parseInt(const agxJson::Value& value) const
      {
        int result = false;

        if (value.isInt())
        {
          result = value.asInt();
        }
        else if (value.isString()) {
          result = atoi(value.asString().c_str());
        }

        return result;
      }

      /**
      Read a bool from a json value.
      */
      bool parseBool( const agxJson::Value& value ) const
      {
        bool result = false;

        if ( value.isBool() )
        {
          result = value.asBool();
        }
        else if ( value.isString() ) {
          result = agx::String(value.asString()).lower() == "true";
        }

        return result;
      }

      /**
      Read a string from a json value.
      */
      agx::String parseString(const agxJson::Value& value) const
      {
        agx::String result;

        if (value.isString())
          result = value.asString();

        return result;
      }

      /**
      Read a real number from a json value
      */
      agx::Real parseReal( const agxJson::Value& value ) const
      {
        if ( value.isDouble() ) {
          return value.asDouble();
        }
        else if ( value.isString() ) {

          if ( value.asString() == "inf" ) {
            return agx::Infinity;
          }
          else if ( value.asString() == "-inf" ) {
            return -agx::Infinity;
          }
        }

        LOGGER_WARNING() << "Unknown value \"" << value.asString() << "\" in json data for " <<
          getParameterPath() << LOGGER_ENDL();

        return 0;
      }


      /**
      Parse a json node and produce a Vec3.
      The result is passed in by reference and should already contain the default values.
      Possible json include:
      - "var" = [1]
      - "var" = [1,2,3]
      - "var" = ["default", 2,3]
      - "var" = "default"
      */
      void parseVec3( const agxJson::Value& value, agx::Vec3& result ) const
      {

        if ( value.isArray() ) {
          if ( value.size() != m_numValues )
            LOGGER_WARNING() << "Array with values for " << m_name << " does not have expected length "
              << m_numValues << "\n" << LOGGER_END();

          agxJson::Value::ArrayIndex maxIndex = std::min(value.size(), std::min(m_numValues, 3u));

          for ( agxJson::Value::ArrayIndex i = 0; i < maxIndex; i++) {
            agxJson::Value valueNode = value[i];

            if ( !isDefault(valueNode) ) {
              result[i] = parseReal( valueNode );
            }
          }

        }
        else if ( value.isDouble() ) {
          // If the value is a single number, then we return a Vec3 with said value
          result = agx::Vec3( value.asDouble() );
        }
        else if ( value.isString() ) {
          if ( !isDefault(value) )
            result = agx::Vec3( parseReal( value ) );
        }
        else {
          LOGGER_WARNING() << "Unknown value \"" << value.asString() << "\" in json data for " <<
            getParameterPath() << LOGGER_ENDL();
        }
      }



      /**
      Reads a generic property from a json structure.
      param root - Pointer to root of data tree
      param material - Pointer to data that will be parsed using a Lambda expression
      param optional - If true, no warnings will be written if m_group or m_name cannot be found in root
      */
      bool readProperty(const agxJson::Value& root, T* material, bool optional=false )
      {
        agxJson::Value object;

        if ( m_group != "" ) {
          if ( !root.hasMember( m_group.c_str() ) ) {
            if (!optional)
              LOGGER_WARNING() << "Json file is missing object " << m_group << LOGGER_ENDL();
            return false;
           }
           object = root[m_group];
        }
        else {
          object = root;
        }

        if ( !object.hasMember( m_name.c_str() ) ) {
          if (!optional)
            LOGGER_WARNING() << "Json object is missing " << m_name << LOGGER_ENDL();
          return false;
        }

        agxJson::Value valueNode = object[m_name];

        // Resolved, the item is found as "valueNode" in the json data.
        // There can still be warnings from the parsing
        m_resolved = true;

        if ( m_type == ARRAY_VEC3 ) {
          // Need to read previous values in case of partial use of default,
          // e.g. [1, "default", 2]
          agx::Vec3 v = m_getVec3( material );

          parseVec3( object[m_name], v );

          m_setVec3( v, material );
        }
        else if ( m_type == SCALAR_REAL) {
          if ( !isDefault(valueNode)) {
            agx::Real r = parseReal( valueNode );
            m_setReal( r, material );
          }
        }
        else if (m_type == STRING) {
          if (!valueNode.asString().empty()) {
            agx::String r = parseString(valueNode);
            m_setString(r, material);
          }
        }
        else if (m_type == SCALAR_INT) {
          if (!isDefault(valueNode)) {
            int val = parseInt(valueNode);
            m_setInt(val, material);
          }
        }
        else if ( m_type == SCALAR_BOOL ) {
          if ( !isDefault(valueNode)) {
            bool b = parseBool( valueNode );
            m_setBool( b, material );
          }
        }
        else {
          LOGGER_WARNING() << "Unknown parameter type for " << getParameterPath() << LOGGER_ENDL();
          return false;
        }

        return true;
      }


      /**
      Writes a real value to a json node
      */
      void writeValueReal( agxJson::Value& node, agx::Real value )
      {
        if ( agx::equivalent(value, agx::Infinity)) {
          node = "inf";
        }
        else if ( agx::equivalent(value, -agx::Infinity)) {
          node = "-inf";
        }
        else {
          node = value;
        }
      }

      /**
      Writes a string value to a json node
      */
      void writeValueString(agxJson::Value& node, const agx::String value)
      {
        node = value;
      }

      /**
      Writes an int value to a json node
      */
      void writeValueInt(agxJson::Value& node, int value)
      {
        node = value;
      }


      /**
      Writes a bool value to a json node
      */
      void writeValueBool( agxJson::Value& node, bool value )
      {
        node = (value ? "true" : "false" );
      }


      /**
      Write a short array, at most 3 elements
      */
      void writeValueVec3( agxJson::Value& node, agx::Vec3 value )
      {
        agxJson::Value::ArrayIndex maxIndex = std::min(m_numValues, 3u);
        node.resize( maxIndex );

        for ( agxJson::Value::ArrayIndex i = 0; i < maxIndex; ++i) {
          node[i] = value[i];
        }
      }


      /**
      Store a generic property to json.
      If group objects are used, then those has to be created in advance.
      */
      bool writeProperty(agxJson::Value& root, const T* material )
      {
        agxJson::Value* outputNode = &root;

        if ( m_group.length() > 0 ) {

          if ( !root.hasMember( m_group ) ) {
            return false;
          }

          outputNode = &root[ m_group ];
        }


        if (m_type == SCALAR_REAL ) {
          writeValueReal( (*outputNode)[ m_name ], m_getReal( material ) );
        }
        else if ( m_type == SCALAR_BOOL ) {
          writeValueBool( (*outputNode)[ m_name ], m_getBool( material ) );
        }
        else if (m_type == SCALAR_INT) {
          writeValueInt((*outputNode)[m_name], m_getInt(material));
        }
        else if (m_type == STRING) {
          writeValueString((*outputNode)[m_name], m_getString(material));
        }
        else if ( m_type == ARRAY_VEC3 ) {
          writeValueVec3( (*outputNode)[ m_name ], m_getVec3( material ) );
        }
        else {
          LOGGER_WARNING() << "Unknown parameter type for " << getParameterPath() << LOGGER_ENDL();
        }

        m_resolved = outputNode->hasMember( m_name );

        return m_resolved;
      }


      /**
      Status flag if parameter could be read/written
      */
      bool isResolved() const
      {
        return m_resolved;
      }

      /**
      Text representation of where in the json structure this parameter lives
      */
      agx::String getParameterPath() const
      {
        if ( m_group.length() > 0 ) {
          return agx::String::format("%s.%s", m_group.c_str(), m_name.c_str());
        }

        return m_name;
      }


    private:
      ParameterType m_type { UNDEFINED_TYPE };
      agx::String   m_name;
      agx::String   m_group;

      unsigned int m_numValues { 0 };

      PropertyGetInt m_getInt;
      PropertySetInt m_setInt;

      PropertyGetBool m_getBool;
      PropertySetBool m_setBool;

      PropertyGetReal m_getReal;
      PropertySetReal m_setReal;

      PropertyGetVec3 m_getVec3;
      PropertySetVec3 m_setVec3;

      PropertyGetString m_getString;
      PropertySetString m_setString;

      bool m_resolved { false };
  };

}

