#pragma once

#include <string>
#include <openplx/TokenType.h>
#include <openplx/Types.h>

namespace openplx
{
    struct Token {
        TokenType type;
        std::string lexeme;
        line_t line;
        col_t column;

        Token() : type(TokenType::Uninitialized), lexeme(), line(0), column(0) {}

#ifdef SWIGCSHARP
        Token(TokenType _type, std::string _lexeme, line_t _line = 0, col_t _column = 0)
          : type(_type), lexeme(std::move(_lexeme)), line(_line), column(_column) {}

#else
        Token(TokenType _type, std::string&& _lexeme, line_t _line = 0, col_t _column = 0)
          : type(_type), lexeme(std::move(_lexeme)), line(_line), column(_column) {}
#endif

        static Token After(TokenType _type, std::string lexeme, const Token& before, col_t offset = 0)
        {
            auto [end_line, end_col] = Token::end_of(before);
            return { _type, std::move(lexeme), end_line, end_col + offset };
        }

        static Token Before(TokenType _type, std::string lexeme, const Token& before, col_t offset = 0)
        {
            return { _type, std::move(lexeme), before.line, before.column - offset - lexeme.size() };
        }

        static Token Uninitialized() {
            return {};
        }
        static Token Invalid() {
            return { TokenType::Invalid, "" };
        }

        static Token Identifier(std::string lexeme) {
            return { TokenType::Identifier, std::move(lexeme) };
        }

        static Token Empty() {
            return { TokenType::NoneType, "" };
        }

        static Token Keyword(TokenType type) {
            switch (type) {
                case TokenType::Is:
                    return { type, "is" };
                case TokenType::As:
                    return { type, "as" };
                case TokenType::Becomes:
                    return { type, "becomes" };
                case TokenType::Static:
                    return { type, "static" };
                case TokenType::Const:
                    return { type, "const" };
                case TokenType::Fn:
                    return { type, "fn" };
                case TokenType::Operator:
                    return { type, "operator" };
                case TokenType::Trait:
                    return { type, "trait" };
                case TokenType::With:
                    return { type, "with" };
                case TokenType::Delete:
                    return { type, "delete" };
                case TokenType::Reference:
                    return { type, "reference" };
                case TokenType::Or:
                    return { type, "or" };
                case TokenType::Optional:
                    return { type, "optional" };
                default:
                    return { type, "" };
            }
        }

        static Token Op(char c) {
            switch (c) {
                case '+':
                    return { TokenType::Plus, "+" };
                case '*':
                    return { TokenType::Star, "*" };
                case '/':
                    return { TokenType::Slash, "/" };
                case '-':
                    return { TokenType::Minus, "-" };
                default:
                    return { TokenType::Uninitialized, "" };
            }
        }


        static Token Bool(bool value) {
            if (value) {
                return { TokenType::TrueKeyword, "true" };
            } else {
                return { TokenType::FalseKeyword, "false" };
            }
        }

        static Token Number(std::string repr) {
            return { TokenType::Number, std::move(repr) };
        }

        static Token Real(double value) {
            return { TokenType::Number, std::to_string(value) };
        }

        static Token Int(int64_t value) {
            return { TokenType::Number, std::to_string(value) };
        }

        static Token String(std::string value) {
            return { TokenType::String, std::move(value) };
        }

        static std::pair<line_t, col_t> end_of(const Token& token) {
            if (token.type == TokenType::MultiLineString) {
                line_t line = token.line;
                col_t column = token.column;
                for (auto& c : token.lexeme) {
                    if (c == '\n') {
                        line++;
                        column = 1;
                    } else {
                        column++;
                    }
                }
                return { line, column };
            } else if (isIdentifierString(token)) {
                return { token.line, token.column + token.lexeme.size() + 2 };
            } else {
                return { token.line, token.column + token.lexeme.size() };
            }
        }

        col_t width() const {
            if (type == TokenType::MultiLineString) {
                return 0;
            } else if (isIdentifierString(*this)) {
                return lexeme.size() + 2;
            } else {
                return lexeme.size();
            }
        }

        static bool isAlpha(char c) {
            return (c >= 'a' && c <= 'z') ||
                (c >= 'A' && c <= 'Z') ||
                c == '_';
        }

        static bool isDigit(char c) {
            return c >= '0' && c <= '9';
        }

        static bool isAlphaNumeric(char c) {
            return isAlpha(c) || isDigit(c);
        }

        static bool isIdentifierString(const Token& token) {
            if (token.type != TokenType::Identifier) return false;
            if (token.lexeme.empty()) return true;
            if (!isAlpha(token.lexeme[0])) return true;
            for (std::size_t i = 1; i < token.lexeme.size(); ++i) {
                if (!isAlphaNumeric(token.lexeme[i])) {
                    return true;
                }
            }
            return false;
        }

        static std::string extractString(const Token& token) {
            if (token.type == TokenType::String) {
                return token.lexeme.substr(1, token.lexeme.size() - 2);
            }
            if (token.type == TokenType::MultiLineString) {
                return token.lexeme.substr(3, token.lexeme.size() - 6);
            }
            if (token.type == TokenType::AtString || token.type == TokenType::DollarString) {
                return token.lexeme.substr(2, token.lexeme.size() - 3);
            }
            return "";
        }

        bool operator==(const Token& other) const {
            return type == other.type &&
                   line == other.line &&
                   column == other.column &&
                   lexeme == other.lexeme;
        }
    };
}
