/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agxTerrain/export.h>
#include <agx/RigidBody.h>
#include <agx/Line.h>
#include <agxCollide/Box.h>
#include <agxUtil/ConvexReaderWriter/ConvexReaderWriter.h>

namespace agxTerrain
{
  class Shovel;

  typedef std::pair<agxCollide::MeshRef, agx::AffineMatrix4x4> MeshTransformPair;
  typedef agx::Vector<MeshTransformPair> MeshTransformPairVector;

  // Surface tangent patch for a specific deadload
  // interval of the shovel.
  struct TangentPatch
  {
    // Specific surface tangent for the patch
    agx::Vec3      m_tangent;
    // Deadload fraction range [0.0-1.0] of patch
    agx::RangeReal m_range;
  };

  using TangentPatchData = agx::Vector<TangentPatch>;

  struct ToolShapeData
  {
    agx::RigidBodyRef shovelBody = nullptr;
    agx::Line topEdge            = agx::Line();
    agx::Line cuttingEdge        = agx::Line();
    agx::Vec3 cuttingDirection   = agx::Vec3();
  };

  struct PointData
  {
    agx::Vec3 point;
    agx::Vec3 normal;
    agx::Real angle;
  };

  struct ShovelSurfaceVectors
  {
    agx::Vec3 averageTangent;
    agx::Vec3 dominantTangent;
    agx::Vec3 averageNormal;
    agx::Vec3 dominantNormal;
    bool valid=false;
  };

  /**
  Struct that holds surface point data gathered through
  surface tracing of the shovel surfaces. Each PointDataVector
  held corresponds to a specific cross-sectional surface trace
  along the shovel surface.
  */
  using PointDataVector = agx::Vector<PointData>;
  struct SurfacePointData
  {
    SurfacePointData() : pointData{} {};

    agx::Vector<PointDataVector> pointData;

    size_t totNumPoints() const {
      size_t s = 0;
      for (auto& v : pointData)
        s += v.size();
      return s;
    }

    bool isValid() const
    {
      // 1. We need to have more than 1 point array returned
      bool valid = pointData.size() > 0;

      // 2. We need at LEAST two points in each array
      for (auto& v : pointData)
        valid &= v.size() > 1;

      return valid;
    }
  };

  struct ShovelPlane
  {
    agx::Vec3 planeDirection;
    agx::Plane plane;
    bool valid = false;
  };

  enum class SURFACE_TRACE_MODE
  {
    INSIDE=0,
    OUTSIDE=1
  };

  /**
  Utility function that creates a custom bucket from a list of box half-vectors and angles
  and associated edges and vectors. The list vectors need to be in the following format:

  [ (Vec3(thicknes, width, length), angle),
    (Vec3(thicknes, width, length), angle),
    ... ].

  The boxes will be placed at the end of each box segement in the [x, z] plane with a
  rotation in the Y-axis specified by the angle element.

  \param sizes - vector of pairs of agx::Vec3 half extents of a box and a rotational angle.
  \param createWalls - true if walls should be created on the side of the bucket geometry, false otherwise.
  \return a ToolShapeData structure with the body, lines and vectors needed to create a shovel.
  */
  AGXTERRAIN_EXPORT ToolShapeData createCustomBucketData( const std::vector<std::pair<agx::Vec3, agx::Real>>& sizes,
                                                          bool createWalls = true );

  /**
  This function traces the inner outline of the specified Shovel's inner body ActiveZone to
  find the internal cross section geometry outline. The returned data is in the local shape
  coordinate system.

  \param shovel - the specified shovel to trace.
  \param numSamplingPoints - the number of points to use in the sampling.
  \param debugRender - true if the outline points should be debug rendered or not.
  */
  AGXTERRAIN_EXPORT std::vector< PointData > traceShovelActiveShapeOutline( agxTerrain::Shovel* shovel,
                                                                            size_t numSamplingPoints = 200,
                                                                            bool debugRender = false );

  /**
  Utility function that creates voxel collision geometries from a parent rigid body and frame.
  Initializes terrainTool <-> terrain voxel collision geometries.

  \param parentBody - Rigid body owning the geometries.
  \param parentFrame - Frame to which new geometries will be attached.
  \param tessellationLevel - Mesh detail level (0.5 = default).
  \return vector of voxel geometry references.
  */
  AGXTERRAIN_EXPORT agxCollide::GeometryRefVector createVoxelCollisionGeometries( agx::RigidBodyRef parentBody,
                                                                                  agx::FrameRef parentFrame,
                                                                                  agx::Real tessellationLevel = 0.5);

  /**
  Trace the surface outline of a shovel with various cross sectional
  ray tracing samples over a specific angle interval, starting with the vector from the
  mesh center to the cutting edge center point, traveling along the shovel surface towards
  the top edge until end angle internval.

  \param shovel - the specific shovel to trace
  \param mode - specify if the traced surface should be inside or outside the shovel (default: OUTSIDE)
  \param numCrossSectionSamples - specify the number of cross sectional slices
  \param numPointsPerSample - specify the number of points per each cross sectional slice
  \param angleInterval - the specific angle interval to use for the tracing for each cross sectional sample
  \param useCutTopAsMidPoint - true if the middle point of the cutting edge <-> top plane should be used as the
                               ray start/end point instead of the mesh center, false otherwise
  \param debugRender - true if debug rendering should be enabled, false otherwise
  \return the SurfacePointData containing all the cross sectional samples
  */
  AGXTERRAIN_EXPORT SurfacePointData traceShovelSurfaceOutline( agxTerrain::Shovel* shovel,
                                                                SURFACE_TRACE_MODE mode = SURFACE_TRACE_MODE::OUTSIDE,
                                                                size_t numCrossSectionSamples = 4,
                                                                size_t numPointsPerSample = 200,
                                                                agx::Real angleInterval = agx::PI,
                                                                bool useCutTopAsMidPoint = false,
                                                                bool debugRender = false );

  AGXTERRAIN_EXPORT bool findPlaneMeshCenterPoint( const agx::Vec3& planePoint,
                                                   const agx::Vec3& planeNormal,
                                                   agx::Real rayLength,
                                                   const MeshTransformPairVector& meshes,
                                                   agx::Vec3& result );

  AGXTERRAIN_EXPORT std::tuple<bool, agx::Vec3> findLineMeshesIntersection( const agx::Vec3& point1,
                                                                            const agx::Vec3& point2,
                                                                            const MeshTransformPairVector& meshes);

  AGXTERRAIN_EXPORT std::tuple<bool, agx::Vec3> findLineShovelIntersection( agxTerrain::Shovel* shovel,
                                                                            const agx::Vec3& point1,
                                                                            const agx::Vec3& point2,
                                                                            bool debugRender = false );




  AGXTERRAIN_EXPORT bool isTriangleFirstInRay(
    const agx::Vec3& bottom,
    const agx::Vec3& top,
    const agxCollide::Mesh::Triangle& triangle,
    const MeshTransformPairVector& meshes );

  /**
  Compute the dominant cutting bottom plane using the backside surface structure.
  Used for determining shovel deformer direction.

  \param shovel - the specific shovel used for computation
  \param debugRender - true if debugrendring should be enabled, false otherwise
  */
  AGXTERRAIN_EXPORT ShovelPlane computeShovelBottomPlane( agxTerrain::Shovel* shovel,
                                                          bool debugRender = false );

  /**
  Compute the dominant inside plane using the inside surface structure.
  Used for determining shovel deformer direction.

  \param shovel - the specific shovel used for computation
  \param debugRender - true if debugrendring should be enabled, false otherwise
  */
  AGXTERRAIN_EXPORT ShovelPlane computeShovelInsidePlane( agxTerrain::Shovel* shovel,
                                                          bool debugRender = false );

  /**
  Compute the surface tangent patches over the whole surface in the angle interval from the
  cutting edge center to the top edge center.

  \param shovel - the specific shovel used for computation
  \param numPoints - the number of points used in total for the surface tracing
  \param numPatches - the number of surface patches that should be used
  \param debugRender - true if debugrendring should be enabled, false otherwise
  */
  AGXTERRAIN_EXPORT TangentPatchData computeInsideTangentPatches( agxTerrain::Shovel* shovel,
                                                                  size_t numPoints = 200,
                                                                  size_t numPatches = 20,
                                                                  bool debugRender = false );

  /**
  Compute the dominant shovel surface vectors such as normal and tangents for a shovel surface
  over a specific angle interval, starting with the vector from the
  mesh center to the cutting edge center point, traveling along the shovel surface towards
  the top edge until end angle internval.

  \param mode - specify if the traced surface should be inside or outside the shovel (default: OUTSIDE)
  \param numCrossSectionSamples - specify the number of cross sectional slices
  \param numPointsPerSample - specify the number of points per each cross sectional slice
  \param angleInterval - the specific angle interval to use for the tracing for each cross sectional sample
  \param debugRender - true if debug rendering should be enabled, false otherwise
  \return the SurfacePointData containing all the cross sectional samples
  */
  AGXTERRAIN_EXPORT ShovelSurfaceVectors computeShovelSurfaceVectors( agxTerrain::Shovel* shovel,
                                                                      SURFACE_TRACE_MODE mode = SURFACE_TRACE_MODE::OUTSIDE,
                                                                      size_t numCrossSectionSamples = 8,
                                                                      size_t numPointsPerSample = 200,
                                                                      agx::Real angleInterval = agx::PI,
                                                                      bool debugRender = false );
}
