/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/TriaxialSignalSystemNode.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( TriaxialSignalScaling );

  /**
  Applies a constant scaling to the triaxial signal.
  */
  class AGXSENSOR_EXPORT TriaxialSignalScaling : public ITriaxialSignalSystemNode
  {
    public:
      /**
      Convenience signal scaling value to convert from the standard AGX Dynamics output SI unit of
      m/s^2 to the commonly occurring accelerometer reading unit of g (as 9.81 m/s^2).
      */
      static constexpr agx::Real CONVERT_METER_PER_SECOND_SQUARED_TO_G = 0.1019368;

      /**
      Convenience signal scaling value to convert from the standard AGX Dynamics output SI unit of
      radians/s to the commonly occurring gyroscope reading unit of degrees/s.
      */
      static constexpr agx::Real CONVERT_RADIANS_PER_SECOND_TO_DEGREES_PER_SECOND = 57.29578;

      /**
      Convenience signal scaling value to convert from the standard AGX Dynamics output SI unit of
      tesla to the commonly occurring magnetometer reading unit of gauss.
      */
      static constexpr agx::Real CONVERT_TESLA_TO_GAUSS = 1e-4;

    public:
      /**
      Construct this signal scaling node using the given \p scaling.
      \param scaling - Sensor signal scaling (unitless)
      */
      TriaxialSignalScaling( const agx::Vec3& scaling = agx::Vec3( 1.0 ) );

      /**
      Set the sensor signal scaling applied by this node.
      \param scaling - Sensor signal scaling (unitless)
      */
      void setScaling( const agx::Vec3& scaling );

      /**
      \return sensor signal scaling (unitless)
      */
      const agx::Vec3& getScaling() const;

      /**
      \return new, unique, triaxial signal system node (proxy) of this implementation
      */
      virtual TriaxialSignalSystemNodeRef createTriaxialNodeProxy() override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void result( SystemNodeProxy& proxy, const CallbackData& data ) override;
      virtual void cleanup( SystemNodeProxy& proxy ) override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::TriaxialSignalScaling );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::Vec3 m_scaling;
  };
}
