/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/agxCore_export.h>
#include <agx/Integer.h>

#include <agx/agx.h>

#include <chrono>



namespace agx {

  /**
  The HighAccurayTimer class can replace the regular Timer class in high level
  applications where the need for accuracy is greater than performance.
  This timer is based upon std::chrono::high_resolution_clock.
  This timer should primarily be used with low calling frequency, such as when
  measuring total computation time.
  */
  class AGXCORE_EXPORT HighAccuracyTimer
  {
  public:

    /**
    Creates a HighAccuracyTimer.
    \param startImmediately Should the HighAccuracyTimer start immediately or start in stopped mode
    */
    HighAccuracyTimer(bool startImmediately = false);

    /**
    Starts the HighAccuracyTimer.
    Calling start on a already started HighAccuracyTimer will have no effect.
    */
    void start();

    /**
    Stops the HighAccuracyTimer.
    Calling stop on a already stopped HighAccuracyTimer will have no effect.
    Upon stopping, the amount of cycles and time ran will be updated.
    */
    void stop();

    /**
    Clear all data and optionally restart the timer.
    */
    void reset(bool startAfterReset = false);

    /**
    Report total elapsed time since start in milliseconds, excluding
    intervals when the timer was suspended.
    */
    Real64 getTime() const;

    /// Return true if running.
    bool isRunning() const;

  private:
    using time_point = std::chrono::high_resolution_clock::time_point;
    using duration   = std::chrono::duration<agx::Real64, std::milli>;

    time_point m_start;
    duration m_total;

    bool m_running;
  };


  /* Implementation */
  inline HighAccuracyTimer::HighAccuracyTimer(bool startImmediately)
  {
    reset();
    if (startImmediately)
      this->start();
  }


  AGX_FORCE_INLINE void HighAccuracyTimer::start()
  {
    if (m_running)
      return;

    m_running = true;

    m_start = std::chrono::high_resolution_clock::now();
  }


  AGX_FORCE_INLINE void HighAccuracyTimer::stop()
  {
    if (m_running) {
      m_running = false;

      time_point end = std::chrono::high_resolution_clock::now();
      duration elapsed = end - m_start;
      m_total += elapsed;
    }
  }


  AGX_FORCE_INLINE void HighAccuracyTimer::reset(bool startAfterReset)
  {
    m_total = duration::zero();

    m_running = false;
    if (startAfterReset)
      this->start();
  }


  AGX_FORCE_INLINE bool HighAccuracyTimer::isRunning() const
  {
    return m_running;
  }


  AGX_FORCE_INLINE Real64 HighAccuracyTimer::getTime() const
  {
    if (m_running)
    {
      time_point now = std::chrono::high_resolution_clock::now();
      duration elapsed = now - m_start;
      duration total = m_total + elapsed;
      return total.count();
    }
    else
    {
      return m_total.count();
    }
  }


} // namespace agx
