#pragma once

/*
Algoryx GPU Sensors API definition.

This header file defines the public API of Algoryx GPU Sensors. These symbols
will be made available as a static library (libAlgoryxGPUSensors.a) that is
built into the consuming library or application. The implementation of these
functions are stubs that loads and calls the corresponding function in the
dynamic library (libAlgoryxGPUSensorsImpl.so) where the actual implementation
lives.
*/

#include "version.h"

#if !defined(AGPU_VERSION_GENERATION)
	#error "AGPU version has not been defined".
#endif


// The the public API minimizes the exposure to Robotec GPU Lidar, some types
// must be made available to the consuming library or application. Example
// types are rgl_status_t and rgl_vec3f.
#include "agpu/api/rgllike_types.h"

#include "agpu/common/types.h"

// Standard library includes.
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>


#ifdef __cplusplus
	#define NO_MANGLING extern "C"
#else // NOT __cplusplus
	#define NO_MANGLING
#endif

#if defined AGPU_STATIC
	#define AGPU_VISIBLE
#else
	#if defined _WIN32
		// Unexpected to not see a dllimport anywhere. Is that not needed?
		#define AGPU_VISIBLE __declspec(dllexport)
	#else
		#define AGPU_VISIBLE __attribute__((visibility("default")))
	#endif
#endif // AGPU_STATIC

#ifndef DEPRECATED
	#if __cplusplus
		#define DEPRECATED(x) [[deprecated(x)]]
	#else // NOT __cplusplus
		#define DEPRECATED(x)
	#endif
#endif // DEPRECATED

#define AGPU_API NO_MANGLING AGPU_VISIBLE


/*********************************** TYPES ************************************/

/**
 * Represents on-GPU shape that can be referenced by Instances in the Scene.
 * Each shape can be referenced by any number of Instances in different Scenes.
 */
typedef struct Shape* agpu_shape_t;

/**
 * Represents an on-GPU material that can be referenced by Instances in the Scene.
 * Each material can be referenced by any number of Instances in different Scenes.
 */
typedef struct Material* agpu_material_t;

/**
 * Opaque handle representing a Scene - a collection of Instances.
 */
typedef struct Scene* agpu_scene_t;

/**
 * Opaque handle representing an object visible to lidars.
 * An Instance is always bound to exactly one Scene.
 */
typedef struct Instance* agpu_instance_t;

/********************************** GENERAL ***********************************/

/**
 * Sets the given paths as additional implementation library search paths when the library is used as a
 * dynamic library.
 * This function does nothing if the library is statically built.
 * This function will not attempt to load the dynamic library.
 * If the dynamic library is already loaded, agpu_cleanup must be called to unload the library before reloading
 * using the given paths is attempted.
 * @param paths Additional search paths. Each path should be a null terminated string.
 * @param path_count Number of additional paths.
 */
AGPU_API agpu_status_code_t agpu_set_additional_library_paths(const char** paths, int32_t path_count);

/**
 * (AlgoryxGPUSensors) Returns data describing semantic version as described in https://semver.org/ with added
 * generation number at the leading end.
 * Version string can be obtained by formatting "{out_generation}.{out_major}.{out_minor}.{out_patch}".
 * Hash is provided mainly for debugging and issue reporting.
 * @param out_generation Address to store generation version number.
 * @param out_major Address to store major version number.
 * @param out_minor Address to store minor version number.
 * @param out_patch Address to store patch version number.
 */
AGPU_API agpu_status_code_t agpu_get_version_info(int32_t* out_generation, int32_t* out_major, int32_t* out_minor,
	int32_t* out_patch);

/**
 * @return True if Algoryx GPU Sensors is currently initialized.
 */
AGPU_API bool agpu_is_initialized();

/**
 * Removes all user-created API objects: Meshes, Instances, Scenes, lidars, etc.
 * Also calls rgl_cleanup().
 * Effectively brings the library to the state as if it was not yet used.
 * All API handles are invalidated.
 */
AGPU_API agpu_status_code_t agpu_cleanup(void);

/**
 * Checks if the passed rgl_status_t is considered unrecoverable.
 * @param status The status to check.
 * @param out_unrecoverable Boolean indicating if passed status is considered unrecoverable.
 */
AGPU_API agpu_status_code_t agpu_is_status_unrecoverable(rgl_status_t status, bool* out_unrecoverable);

/**
 * Get a human-readable string describing the most recent error that was
 * detected within Algoryx GPU Sensors. This is separate from
 * agpu_rgl_get_last_error_string, which returns an error string from Robotec
 * GPU Lidar.
 */
AGPU_API void agpu_get_last_error_string(const char** out_error_string);


/*********************************** DEVICE ***********************************/


/**
 * Lists the available compute devices.
 * This function will not initialize the library, but will attempt to load the dynamic library if AlgoryxGPUSensors
 * has not been built statically.
 * @param count Address where to store the number of available compute devices.
 * @param names Address to an array of char* where to store the names of the compute devices. (Can be NULL)
 */
AGPU_API agpu_status_code_t agpu_device_list_available(int32_t* count, char** names);

/**
 * Returns the index of the compute device currently in use by AlgoryxGPUSensors.
 * @param device_index Address where to store the current device index.
 */
AGPU_API agpu_status_code_t agpu_device_get_current(int32_t* device_index);

/**
 * Attempts to set the compute device which AlgoryxGPUSensors should use.
 * WARNING! This function will call agpu_cleanup() before attempting to set the compute device. All resources
 * will be deallocated in the process! Make sure to manage any dangling handles this may cause.
 * @param device_index Index of the compute device which AlgoryxGPUSensors should attempt to use.
 */
AGPU_API agpu_status_code_t agpu_device_set_current(int32_t device_index);

/*********************************** SHAPE ************************************/

/**
 * Creates Mesh (Shape) from the arrays of vertices and indices. CW/CCW order matters.
 * Provided arrays are copied to the GPU before this function returns.
 * @param out_mesh Address to store the resulting Mesh handle.
 * @param vertices An array of rgl_vec3f or binary-compatible data representing Mesh vertices.
 * @param vertex_count Number of elements in the vertices array.
 * @param indices An array of rgl_vec3i or binary-compatible data representing Mesh indices.
 * @param index_count Number of elements in the indices array.
 * @param normals An array of rgl_vec3f or binary-compatible data representing a normal per vertex. (Can be NULL)
 * @param uv An array of rgl_vec2f or binary-compatible data representing a UV coordinate per vertex. (Can be NULL)
 */
AGPU_API agpu_status_code_t agpu_shape_mesh_create(agpu_shape_t* out_mesh, const rgl_vec3f* vertices,
	int32_t vertex_count, const rgl_vec3i* indices, int32_t index_count, const rgl_vec3f* normals, const rgl_vec2f* uv);

/**
 * Updates Mesh (Shape) vertex data. The number of vertices must not change.
 * This function is intended to update animated meshes.
 * @param mesh Mesh (Shape) to modify
 * @param vertices An array of rgl_vec3f or binary-compatible data representing Mesh vertices
 * @param vertex_count Number of elements in the vertices array. It must be equal to the original vertex count!
 */
AGPU_API agpu_status_code_t agpu_shape_mesh_update_vertices(agpu_shape_t mesh, const rgl_vec3f* vertices,
	int32_t vertex_count);

/**
 * Creates a Sphere (Shape) with the given radius.
 * @param out_sphere Address to store the resulting Sphere (Shape) handle.
 * @param radius Sphere radius.
 */
AGPU_API agpu_status_code_t agpu_shape_sphere_create(agpu_shape_t* out_sphere, float radius);

/**
 * Updates the Sphere (Shape) radius.
 * @param sphere Sphere (Shape) to modify.
 * @param radius Radius of the sphere.
 */
AGPU_API agpu_status_code_t agpu_shape_sphere_update_radius(agpu_shape_t sphere, float radius);

/**
 * Informs that the given Shape will be no longer used.
 * The Shape will be destroyed after all referring Instances are destroyed.
 * @param shape Mesh to be marked as no longer needed.
 */
AGPU_API agpu_status_code_t agpu_shape_destroy(agpu_shape_t shape);

/**
 * Assigns value true to the out_alive flag if the given Shape is known and has not been destroyed, assigns value
 * false otherwise.
 * @param shape Shape to check if alive
 * @param out_alive Boolean set to indicate if alive
 */
AGPU_API agpu_status_code_t agpu_shape_is_alive(agpu_shape_t shape, bool* out_alive);

/********************************* MATERIAL ***********************************/

/**
 * Creates a homogeneous atmospheric material.
 * @param out_material Address to store the resulting Material handle.
 * @param n Refractive index of the atmosphere. If 0, no refraction will occur.
 * @param alpha Attenuation coefficient.
 * @param A Atmospheric return gamma-distribution probability scaling. If 0, no returns are produced.
 * @param k Atmospheric return gamma-distribution shape parameter. Should be > 0.
 * @param theta Atmospheric return gamma-distribution scale parameter. Should be > 0.
 */
AGPU_API agpu_status_code_t agpu_material_atmosphere_create(agpu_material_t* out_material, float n, float alpha,
	float A, float k, float theta);

/**
 * Fetches the current parameters for the given atmospheric material.
 * @param material Material whose parameters to get.
 * @param n Address where to store the atmosphere's refractive index. (Can be NULL)
 * @param alpha Address where to store the attenuation coefficient. (Can be NULL)
 * @param A Address where to store the return distribution probability scaling. (Can be NULL)
 * @param k Address where to store the return distribution shape parameter. (Can be NULL)
 * @param theta Address where to store the return distribution scale parameter. (Can be NULL)
 */
AGPU_API agpu_status_code_t agpu_material_atmosphere_get_parameters(agpu_material_t material, float* n, float* alpha,
	float* A, float* k, float* theta);

/**
 * Updates the atmosphere material parameters.
 * @param material Material whose parameters to update.
 * @param n Refractive index of the atmosphere. If 0, no refraction will occur.
 * @param alpha Attenuation coefficient.
 * @param A Atmospheric return gamma-distribution probability scaling. If 0, no returns are produced.
 * @param k Atmospheric return gamma-distribution shape parameter. Should be > 0.
 * @param theta Atmospheric return gamma-distribution scale parameter. Should be > 0.
 */
AGPU_API agpu_status_code_t agpu_material_atmosphere_update_parameters(agpu_material_t material, float n, float alpha,
	float A, float k, float theta);

/**
 * Creates an explicitly defined opaque BRDF material. Any non-reflected light is absorbed.
 * @param out_material Address to store the resulting Material handle.
 * @param samples Array containing the BRDF samples for the specific wavelength of the raytrace. Samples should be
 *  arranged as a flattened fourth order tensor with parameters in the order `S(thetaL, phiL, thetaV, phiV)`, where the
 *  last index is flattened without stride.
 * @param resThetaL Vertical light sample resolution.
 * @param resPhiL Azimuthal light sample resolution.
 * @param resThetaV Vertical view sample resolution.
 * @param resPhiV Azimuthal view sample resolution.
 */
AGPU_API agpu_status_code_t agpu_material_explicit_brdf_opaque_create(agpu_material_t* out_material,
	const float* samples, unsigned int resThetaL, unsigned int resPhiL, unsigned int resThetaV, unsigned int resPhiV);

/**
 * Fetches the current parameters for the given explicit BRDF material.
 * @param material Material whose parameters to get.
 * @param samples Address to an array of size `resThetaL * resPhiL * resThetaV * resPhiV` where to store the explicit
 *  BRDF samples. (Can be NULL)
 * @param resThetaL Address where to store the vertical light sample resolution. (Can be NULL)
 * @param resPhiL Address where to store the azimuthal light sample resolution. (Can be NULL)
 * @param resThetaV Address where to store the vertical view sample resolution. (Can be NULL)
 * @param resPhiV Address where to store the azimuthal view sample resolution. (Can be NULL)
 */
AGPU_API agpu_status_code_t agpu_material_explicit_brdf_opaque_get_parameters(agpu_material_t material, float* samples,
	unsigned int* resThetaL, unsigned int* resPhiL, unsigned int* resThetaV, unsigned int* resPhiV);

/**
 * Updates the atmosphere material parameters.
 * @param material Material whose parameters to update.
 * @param samples Array containing the BRDF samples for the specific wavelength of the raytrace. Samples should be
 *  arranged as a flattened fourth order tensor with parameters in the order `S(thetaL, phiL, thetaV, phiV)`, where the
 *  last index is flattened without stride.
 * @param resThetaL Vertical light sample resolution.
 * @param resPhiL Azimuthal light sample resolution.
 * @param resThetaV Vertical view sample resolution.
 * @param resPhiV Azimuthal view sample resolution.
 */
AGPU_API agpu_status_code_t agpu_material_explicit_brdf_opaque_update_parameters(agpu_material_t material,
	const float* samples, unsigned int resThetaL, unsigned int resPhiL, unsigned int resThetaV, unsigned int resPhiV);

/**
 * Creates an opaque GGX and Oren-Nayar two-layer material. Any non-reflected light is absorbed.
 * @param out_material Address to store the resulting Material handle.
 * @param n Refractive index real part.
 * @param k Refractive index imaginary part. (0 for non-absorbing surfaces)
 * @param bRoughness Beckman roughness of the top-layer.
 * @param onRoughness Oren-Nayar roughness of the diffuse layer. (Can be related to Beckman roughness as
 *	Oren-Nayar roughness = 1/sqrt(2) arctan(Beckman roughness) )
 * @param diffuseReflectance Base reflectance scaling of the diffuse secondary layer.
 */
AGPU_API agpu_status_code_t agpu_material_ggx_oren_nayar_opaque_create(agpu_material_t* out_material, float n, float k,
	float bRoughness, float onRoughness, float diffuseReflectance);

/**
 * Fetches the current parameters for the given GGX and Oren-Nayar two-layer  material.
 * @param n Address where to store the refractive index real part value. (Can be NULL)
 * @param k Address where to store the refractive index imaginary part value. (Can be NULL)
 * @param bRoughness Address where to store the Beckman roughness value. (Can be NULL)
 * @param onRoughness Address where to store the Oren-Nayar roughness value. (Can be NULL)
 * @param diffuseReflectance Address where to store the diffuse layer base reflectance value. (Can be NULL)
 */
AGPU_API agpu_status_code_t agpu_material_ggx_oren_nayar_opaque_get_parameters(agpu_material_t material, float* n,
	float* k, float* bRoughness, float* onRoughness, float* diffuseReflectance);

/**
 * Updates the opaque GGX and Oren-Nayar two-layer material parameters.
 * @param material Material whose parameters to update.
 * @param n Refractive index real part.
 * @param k Refractive index imaginary part. (0 for non-absorbing surfaces)
 * @param bRoughness Beckman roughness of the top-layer.
 * @param onRoughness Oren-Nayar roughness of the diffuse layer. (Can be related to Beckman roughness as
 *	Oren-Nayar roughness = 1/sqrt(2) arctan(Beckman roughness) )
 * @param diffuseReflectance Base reflectance scaling of the diffuse secondary layer.
 */
AGPU_API agpu_status_code_t agpu_material_ggx_oren_nayar_opaque_update_parameters(agpu_material_t material, float n,
	float k, float bRoughness, float onRoughness, float diffuseReflectance);

/**
 * Creates a thin transparent material with a GGX surface.
 * Any light that is not reflected or absorbed at the surface is transmitted.
 * @param out_material Address to store the resulting Material handle.
 * @param n Refractive index real part.
 * @param k Refractive index imaginary part. (0 for non-absorbing surfaces)
 * @param bRoughness Beckman roughness of the top-layer.
 * @param thickness Thickness of the thin glass.
 */
AGPU_API agpu_status_code_t agpu_material_ggx_thin_glass_create(agpu_material_t* out_material, float n, float k,
	float bRoughness, float thickness);

/**
 * Fetches the current parameters for the thin transparent material with a GGX surface.
 * @param n Address where to store the refractive index real part value. (Can be NULL)
 * @param k Address where to store the refractive index imaginary part value. (Can be NULL)
 * @param bRoughness Address where to store the Beckman roughness value. (Can be NULL)
 * @param thickness Address where to store the thickness value. (Can be NULL)
 * @param diffuseReflectance Address where to store the diffuse layer base reflectance value. (Can be NULL)
 */
AGPU_API agpu_status_code_t agpu_material_ggx_thin_glass_get_parameters(agpu_material_t material, float* n, float* k,
	float* bRoughness, float* thickness);

/**
 * Updates the thin transparent material with a GGX surface.
 * @param material Material whose parameters to update.
 * @param n Refractive index real part.
 * @param k Refractive index imaginary part. (0 for non-absorbing surfaces)
 * @param bRoughness Beckman roughness of the top-layer.
 * @param thickness Thickness of the thin glass.
 */
AGPU_API agpu_status_code_t agpu_material_ggx_thin_glass_update_parameters(agpu_material_t material, float n, float k,
	float bRoughness, float thickness);

/**
 * Creates an opaque Lambertian scatterer material with the given reflectance. Any non-reflected light is absorbed.
 * @param out_material Address to store the resulting Material handle.
 * @param reflectance Diffuse reflectance value (factor amount of light reflected at normal incidence). Any non-
 * reflected light is absorbed.
 */
AGPU_API agpu_status_code_t agpu_material_lambertian_opaque_create(agpu_material_t* out_material, float reflectance);

/**
 * Fetches the current parameters for the given Lambertian scatterer material.
 * @param reflectance Address where to store the reflectance value. (Can be NULL)
 */
AGPU_API agpu_status_code_t agpu_material_lambertian_opaque_get_parameters(agpu_material_t material,
	float* reflectance);

/**
 * Updates the opaque Lambertian scatterer material parameters.
 * @param material Material whose parameters to update.
 * @param reflectance Diffuse reflectance value (factor amount of light reflected at normal incidence). Any non-
 * reflected light is absorbed.
 */
AGPU_API agpu_status_code_t agpu_material_lambertian_opaque_update_parameters(agpu_material_t material,
	float reflectance);

/**
 * Informs that the given Material will be no longer used.
 * The Material will be destroyed after all referring Instances are destroyed.
 * @param material Material to be marked as no longer needed.
 */
AGPU_API agpu_status_code_t agpu_material_destroy(agpu_material_t material);

/**
 * Assigns value true to the out_alive flag if the given Material is known and has not been destroyed, assigns value
 * false otherwise.
 * @param material Material to check if alive
 * @param out_alive Boolean set to indicate if alive
 */
AGPU_API agpu_status_code_t agpu_material_is_alive(agpu_material_t material, bool* out_alive);

/********************************** SCENE *************************************/

/**
 * Creates a new Scene, effectively a world where instances are housed.
 * @param out_scene Address to where to store the resulting scene.
 */
AGPU_API agpu_status_code_t agpu_scene_create(agpu_scene_t* out_scene);

/**
 * Configures the ambient medium of the scene.
 * This is the medium in which any ray will start its tracing. Default is no medium (NULL).
 * @param scene Scene to configure.
 * @param medium Ambient medium to add to the scene (NULL removes any previously set ambient medium).
 */
AGPU_API agpu_status_code_t agpu_scene_set_ambient_medium(agpu_scene_t scene, agpu_material_t medium);

/**
 * Informs that the given Scene will be no longer used.
 * @param scene Scene to be destroyed.
 */
AGPU_API agpu_status_code_t agpu_scene_destroy(agpu_scene_t scene);

/**
 * Assigns value true to the out_alive flag if the given Scene is known and has not been destroyed, assigns value
 * false otherwise.
 * @param scene Scene to check if alive
 * @param out_alive Boolean set to indicate if alive
 */
AGPU_API agpu_status_code_t agpu_scene_is_alive(agpu_scene_t scene, bool* out_alive);

/********************************* INSTANCE ***********************************/

/**
 * Creates a new Instance in the given Scene.
 * @param out_instance Handle to the created Instance.
 * @param scene Scene where the Instance will be added.
 */
AGPU_API agpu_status_code_t agpu_instance_create(agpu_instance_t* out_instance, agpu_scene_t scene);

/**
 * Links the specified Shape to the given Instance.
 * @param instance Instance modify.
 * @param shape Mesh to link to the instance.
 */
AGPU_API agpu_status_code_t agpu_instance_set_shape(agpu_instance_t instance, agpu_shape_t shape);

/**
 * Links the specified Material to the given Instance.
 * @param instance Instance modify.
 * @param material Material to link to the instance.
 */
AGPU_API agpu_status_code_t agpu_instance_set_material(agpu_instance_t instance, agpu_material_t material);

/**
 * Changes transform (position, rotation, scaling) of the given Instance.
 * @param instance Instance to modify
 * @param transform Pointer to rgl_mat3x4f (or binary-compatible data) representing desired (Instance -> world) coordinate
 * system transform
 */
AGPU_API agpu_status_code_t agpu_instance_set_pose(agpu_instance_t instance, const rgl_mat3x4f* transform);

/**
 * Set the user supplied entity ID of the given Instance.
 * @param instance Instance to set id of
 * @param entity_id Integer set to the entity id
 */
AGPU_API agpu_status_code_t agpu_instance_set_entity_id(agpu_instance_t instance, int32_t entity_id);

/**
 * Removes an Instance from the Scene and releases its resources (memory).
 * This operation does not affect the Instance's Mesh since it can be shared among other Instances
 * @param instance Instance to remove
 */
AGPU_API agpu_status_code_t agpu_instance_destroy(agpu_instance_t instance);

/**
 * Assigns value true to the out_alive flag if the given Instance is known and has not been destroyed, assigns value
 * false otherwise.
 * @param instance Instance to check if alive
 * @param out_alive Boolean set to indicate if alive
 */
AGPU_API agpu_status_code_t agpu_instance_is_alive(agpu_instance_t instance, bool* out_alive);

/*********************************** NODES ************************************/

/**
 * Creates or modifies BeamDivergenceAttenuationNode.
 * The Node attenuates the hit intensity based on the optical path length (distance) to the hit.
 * Graph input: ray hits
 * Graph output: ray hits
 * @param node If (*node) == nullptr, a new Node will be created. Otherwise, (*node) will be modified.
 * @param divergence Beam divergence (half-) angle in radians (between 0 and 0.5*PI).
 * @param E Emitter radius (Zero indicates a pure unobstructed Gaussian beam).
 * @param R Detector radius.
 * @param rD Detector radial displacement from the emitter (Zero indicates collinear emitter and detector).
 */
AGPU_API agpu_status_code_t agpu_node_beam_divergence_attenuation(rgl_node_t* node, float divergence, float E, float R,
	float rD);

/**
 * Creates or modifies BeamDivergencePointAttenuationNode.
 * The Node attenuates the return intensity based on the optical path length (distance) to the return.
 * Graph input: point cloud
 * Graph output: point cloud
 * @param node If (*node) == nullptr, a new Node will be created. Otherwise, (*node) will be modified.
 * @param divergence Beam divergence (half-) angle in radians (between 0 and 0.5*PI).
 * @param E Emitter radius (Zero indicates a pure unobstructed Gaussian beam).
 * @param R Detector radius.
 * @param rD Detector radial displacement from the emitter (Zero indicates collinear emitter and detector).
 */
AGPU_API agpu_status_code_t agpu_node_beam_divergence_point_attenuation(rgl_node_t* node, float divergence, float E, float R,
	float rD);

/**
 * Creates or modifies DiscretePointAssemblyNode.
 * The Node reprocesses ray hits from the (PEXT_RGL) raytrace node into a point cloud, as one point per ray.
 * Graph input: ray hits
 * Graph output: point cloud
 * @param node If (*node) == nullptr, a new Node will be created. Otherwise, (*node) will be modified.
 * @param atmospheric_hit_seed Seed used for the pseudo-random generation of atmospheric hits.
 */
AGPU_API agpu_status_code_t agpu_node_discrete_point_assembly(rgl_node_t* node, unsigned int atmospheric_hit_seed);

/**
 * Creates or modifies MultistepRaytraceNode.
 * The Node performs multi-step GPU-accelerated raytracing in the given Scene.
 * NOTE:
 * Output memory usage scales exponentially with each additional raytrace step.
 * Graph input: rays
 * Graph output: ray hits
 * @param node If (*node) == nullptr, a new Node will be created. Otherwise, (*node) will be modified.
 * @param scene Scene to perform raytracing in.
 * @param traces Number of raytracing steps.
 */
AGPU_API agpu_status_code_t agpu_node_raytrace_multistep(rgl_node_t* node, agpu_scene_t scene,
	uint32_t traces);

/**
 * Creates or modifies RandomIntensityFilter.
 * The Node randomly filters out nodes as misses based on the intensity as `a + pow(1 - (b * I), c)` for an intensity I.
 * NOTE: Does not compact the point cloud.
 * Graph input: point cloud
 * Graph output: point cloud
 * @param node If (*node) == nullptr, a new Node will be created. Otherwise, (*node) will be modified.
 * @param a Intensity offset value.
 * @param b Pre-exponentiation intensity scaling value.
 * @param c Intensity exponentiation value.
 * @param point_drop_seed Seed used for the pseudo-random dropping of points.
 */
AGPU_API agpu_status_code_t agpu_node_random_intensity_filter(rgl_node_t* node, const float a, const float b, const float c,
	unsigned int point_drop_seed);

/**
 * Creates or modifies RayAngelGaussianNoise.
 * The Node applies random Gaussian perturbations to the ray, in its local frame, around the given axis.
 * Graph input: rays
 * Graph output: rays
 * @param node If (*node) == nullptr, a new Node will be created. Otherwise, (*node) will be modified.
 * @param mean Mean of the Gaussian distribution (in radians).
 * @param std Standard deviation of the Gaussian distribution (in radians).
 * @param axis Axis in ray local frame to rotate around.
 * @param angle_noise_seed Seed used for the pseudo-random angular perturbation.
 */
AGPU_API agpu_status_code_t agpu_node_ray_angle_gaussian_noise(rgl_node_t* node, const float mean, const float std,
	const rgl_vec3f axis, unsigned int angle_noise_seed);

/*********************************** GRAPH ************************************/

/**
 * Synchronizes this thread with the execution of the graph containing the given Node.
 * This function will block until the graph execution is finished.
 * @param node Any Node from the graph to synchronize with.
 */
AGPU_API agpu_status_code_t agpu_graph_synchronize(rgl_node_t node);

/*
 RGL stubs
*/

AGPU_API agpu_status_code_t agpu_rgl_get_version_info(int32_t *out_major, int32_t *out_minor, int32_t *out_patch);
AGPU_API agpu_status_code_t agpu_rgl_get_extension_info(rgl_extension_t extension, int32_t *out_available);
AGPU_API agpu_status_code_t agpu_rgl_configure_logging(rgl_log_level_t log_level, const char *log_file_path,
	bool use_stdout);
AGPU_API void agpu_rgl_get_last_error_string(const char **out_error_string);
AGPU_API agpu_status_code_t agpu_rgl_cleanup(void);
AGPU_API agpu_status_code_t agpu_rgl_mesh_create(rgl_mesh_t *out_mesh, const rgl_vec3f *vertices, int32_t vertex_count,
	const rgl_vec3i *indices, int32_t index_count);
AGPU_API agpu_status_code_t agpu_rgl_mesh_set_texture_coords(rgl_mesh_t mesh, const rgl_vec2f *uvs, int32_t uv_count);
AGPU_API agpu_status_code_t agpu_rgl_mesh_destroy(rgl_mesh_t mesh);
AGPU_API agpu_status_code_t agpu_rgl_mesh_update_vertices(rgl_mesh_t mesh, const rgl_vec3f *vertices,
	int32_t vertex_count);
AGPU_API agpu_status_code_t agpu_rgl_mesh_is_alive(rgl_mesh_t mesh, bool *out_alive);
AGPU_API agpu_status_code_t agpu_rgl_entity_create(rgl_entity_t *out_entity, rgl_scene_t scene, rgl_mesh_t mesh);
AGPU_API agpu_status_code_t agpu_rgl_entity_destroy(rgl_entity_t entity);
AGPU_API agpu_status_code_t agpu_rgl_entity_set_pose(rgl_entity_t entity, const rgl_mat3x4f *transform);
AGPU_API agpu_status_code_t agpu_rgl_entity_set_id(rgl_entity_t entity, int32_t id);
AGPU_API agpu_status_code_t agpu_rgl_entity_set_intensity_texture(rgl_entity_t entity, rgl_texture_t texture);
AGPU_API agpu_status_code_t agpu_rgl_entity_set_laser_retro(rgl_entity_t entity, float retro);
AGPU_API agpu_status_code_t agpu_rgl_entity_is_alive(rgl_entity_t entity, bool *out_alive);
AGPU_API agpu_status_code_t agpu_rgl_texture_create(rgl_texture_t *out_texture, const void *texels, int32_t width,
	int32_t height);
AGPU_API agpu_status_code_t agpu_rgl_texture_destroy(rgl_texture_t texture);
AGPU_API agpu_status_code_t agpu_rgl_texture_is_alive(rgl_texture_t texture, bool *out_alive);
AGPU_API agpu_status_code_t agpu_rgl_scene_set_time(rgl_scene_t scene, uint64_t nanoseconds);
AGPU_API agpu_status_code_t agpu_rgl_node_rays_from_mat3x4f(rgl_node_t *node, const rgl_mat3x4f *rays,
	int32_t ray_count);
AGPU_API agpu_status_code_t agpu_rgl_node_rays_set_ring_ids(rgl_node_t *node, const int32_t *ring_ids,
	int32_t ring_ids_count);
AGPU_API agpu_status_code_t agpu_rgl_node_rays_set_range(rgl_node_t *node, const rgl_vec2f *ranges,
	int32_t ranges_count);
AGPU_API agpu_status_code_t agpu_rgl_node_rays_set_time_offsets(rgl_node_t *node, const float *offsets,
	int32_t offsets_count);
AGPU_API agpu_status_code_t agpu_rgl_node_rays_transform(rgl_node_t *node, const rgl_mat3x4f *transform);
AGPU_API agpu_status_code_t agpu_rgl_node_points_transform(rgl_node_t *node, const rgl_mat3x4f *transform);
AGPU_API agpu_status_code_t agpu_rgl_node_raytrace(rgl_node_t *node, rgl_scene_t scene);
AGPU_API agpu_status_code_t agpu_rgl_node_raytrace_configure_velocity(rgl_node_t node,
	const rgl_vec3f *linear_velocity, const rgl_vec3f *angular_velocity);
AGPU_API agpu_status_code_t agpu_rgl_node_raytrace_configure_distortion(rgl_node_t node, bool enable);
AGPU_API agpu_status_code_t agpu_rgl_node_raytrace_configure_non_hits(rgl_node_t node, float nearDistance,
	float farDistance);
AGPU_API agpu_status_code_t agpu_rgl_node_raytrace_configure_mask(rgl_node_t node, const int8_t *rays_mask,
	int32_t rays_count);
AGPU_API agpu_status_code_t agpu_rgl_node_raytrace_configure_beam_divergence(rgl_node_t node,
	float horizontal_beam_divergence, float vertical_beam_divergence);
AGPU_API agpu_status_code_t agpu_rgl_node_raytrace_configure_default_intensity(rgl_node_t node,
	float default_intensity);
AGPU_API agpu_status_code_t agpu_rgl_node_points_format(rgl_node_t *node, const rgl_field_t *fields,
	int32_t field_count);
AGPU_API agpu_status_code_t agpu_rgl_node_points_yield(rgl_node_t *node, const rgl_field_t *fields,
	int32_t field_count);
AGPU_API agpu_status_code_t agpu_rgl_node_points_compact_by_field(rgl_node_t *node, rgl_field_t field);
AGPU_API agpu_status_code_t agpu_rgl_node_points_spatial_merge(rgl_node_t *node, const rgl_field_t *fields,
	int32_t field_count);
AGPU_API agpu_status_code_t agpu_rgl_node_points_temporal_merge(rgl_node_t *node, const rgl_field_t *fields,
	int32_t field_count);
AGPU_API agpu_status_code_t agpu_rgl_node_points_from_array(rgl_node_t *node, const void *points, int32_t points_count,
	const rgl_field_t *fields, int32_t field_count);
AGPU_API agpu_status_code_t agpu_rgl_node_points_radar_postprocess(rgl_node_t *node,
	const rgl_radar_scope_t *radar_scopes, int32_t radar_scopes_count, float ray_azimuth_step, float ray_elevation_step,
	float frequency, float power_transmitted, float cumulative_device_gain, float received_noise_mean,
	float received_noise_st_dev);
AGPU_API agpu_status_code_t agpu_rgl_node_points_radar_track_objects(rgl_node_t *node, float object_distance_threshold,
	float object_azimuth_threshold, float object_elevation_threshold, float object_radial_speed_threshold,
	float max_matching_distance, float max_prediction_time_frame, float movement_sensitivity);
AGPU_API agpu_status_code_t agpu_rgl_node_points_radar_set_classes(rgl_node_t node, const int32_t *entity_ids,
	const rgl_radar_object_class_t *object_classes, int32_t count);
AGPU_API agpu_status_code_t agpu_rgl_node_points_filter_ground(rgl_node_t *node, const rgl_vec3f *sensor_up_vector,
	float ground_angle_threshold);
AGPU_API agpu_status_code_t agpu_rgl_node_gaussian_noise_angular_ray(rgl_node_t *node, float mean, float st_dev,
	rgl_axis_t rotation_axis, unsigned int seed);
AGPU_API agpu_status_code_t agpu_rgl_node_gaussian_noise_angular_hitpoint(rgl_node_t *node, float mean, float st_dev,
	rgl_axis_t rotation_axis, unsigned int seed);
AGPU_API agpu_status_code_t agpu_rgl_node_gaussian_noise_distance(rgl_node_t *node, float mean, float st_dev_base,
	float st_dev_rise_per_meter, unsigned int seed);
AGPU_API agpu_status_code_t agpu_rgl_node_multi_return_switch(rgl_node_t *node, rgl_return_type_t return_type);
AGPU_API agpu_status_code_t agpu_rgl_node_is_alive(rgl_node_t node, bool *out_alive);
AGPU_API agpu_status_code_t agpu_rgl_graph_run(rgl_node_t node);
AGPU_API agpu_status_code_t agpu_rgl_graph_destroy(rgl_node_t node);
AGPU_API agpu_status_code_t agpu_rgl_graph_get_result_size(rgl_node_t node, rgl_field_t field, int32_t *out_count,
	int32_t *out_size_of);
AGPU_API agpu_status_code_t agpu_rgl_graph_get_result_data(rgl_node_t node, rgl_field_t field, void *data);
AGPU_API agpu_status_code_t agpu_rgl_graph_node_add_child(rgl_node_t parent, rgl_node_t child);
AGPU_API agpu_status_code_t agpu_rgl_graph_node_remove_child(rgl_node_t parent, rgl_node_t child);
AGPU_API agpu_status_code_t agpu_rgl_graph_node_set_priority(rgl_node_t node, int32_t priority);
AGPU_API agpu_status_code_t agpu_rgl_graph_node_get_priority(rgl_node_t node, int32_t *out_priority);
