/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agxTerrain/export.h>
#include <agx/RigidBody.h>
#include <agx/Line.h>
#include <agxCollide/Box.h>
#include <agxUtil/ConvexReaderWriter/ConvexReaderWriter.h>

namespace agxTerrain
{
  class Shovel;

  struct ToolShapeData
  {
    agx::RigidBodyRef shovelBody = nullptr;
    agx::Line topEdge            = agx::Line();
    agx::Line cuttingEdge        = agx::Line();
    agx::Vec3 cuttingDirection   = agx::Vec3();
  };

  struct PointData
  {
    agx::Vec3 point;
    agx::Vec3 normal;
    agx::Real angle;
  };

  /**
  Utility function that creates a custom bucket from a list of box half-vectors and angles
  and associated edges and vectors. The list vectors need to be in the following format:

  [ (Vec3(thicknes, width, length), angle),
    (Vec3(thicknes, width, length), angle),
    ... ].

  The boxes will be placed at the end of each box segement in the [x, z] plane with a
  rotation in the Y-axis specified by the angle element.

  \param sizes - vector of pairs of agx::Vec3 half extents of a box and a rotational angle.
  \param createWalls - true if walls should be created on the side of the bucket geometry, false otherwise.
  \return a ToolShapeData structure with the body, lines and vectors needed to create a shovel.
  */
  AGXTERRAIN_EXPORT ToolShapeData createCustomBucketData( const std::vector<std::pair<agx::Vec3, agx::Real>>& sizes,
                                                          bool createWalls = true );

  /**
  This function traces the inner outline of the specified Shovel's inner body ActiveZone to
  find the internal cross section geometry outline. The returned data is in the local shape
  coordinate system.

  \param shovel - the specified shovel to trace.
  \param numSamplingPoints - the number of points to use in the sampling.
  \param debugRender - true if the outline points should be debug rendered or not.
  */
  AGXTERRAIN_EXPORT std::vector< PointData > traceShovelActiveShapeOutline( agxTerrain::Shovel* shovel,
                                                                            size_t numSamplingPoints = 200,
                                                                            bool debugRender = false );
}

