/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agx/config/AGX_USE_AGXTERRAIN.h>
#include <agxTerrain/export.h>

#include <array>
#include <agx/Vec3.h>
#include <agx/agx_vector_types.h>

#include <agxTerrain/FailureSurface.h>
#include <agxTerrain/FailureSurfaceAggregate.h>
#include <agxTerrain/GroundCollapseUtils.h>

namespace agxTerrain
{
  namespace GroundCollapse
  {
    /*
Result from a solved contact in the ground collapse simulation.
*/
    struct AGXTERRAIN_EXPORT ContactResult
    {
      agx::Vec3 localPoint;
      agx::Vec3 localForce;
      agx::Vec3f normal;
      agx::Vec3f tangentU;
      agx::Vec3f tangentV;
      agx::Real fos;

      agx::Vec3 getNormalForce() const
      {
        return static_cast<agx::Vec3>(normal) * localForce[ 0 ];
      }

      agx::Vec3 getFrictionForce() const
      {
        return static_cast<agx::Vec3>(tangentU) * localForce[ 1 ]
          + static_cast<agx::Vec3>(tangentV) * localForce[ 2 ];
      }

      ~ContactResult() = default;
    };

    using ContactResultVector = agx::VectorPOD<ContactResult>;

    /*
    Solve result for a substep in the ground collapse algorithm.
    Contains the contact solve results, as well as aggregate
    properties.
    */
    struct AGXTERRAIN_EXPORT SubStepResult
    {
      agx::UInt             subStep = 0;
      ContactResultVector   contactResults;
      agx::Vec3             aggregatePosition;
      agx::Vec3             aggregateAcceleration;
      agx::Vec3             aggregateVelocity;
      /*Factor of safety*/
      agx::Real             fos = 0.0;
      /*Weighted Factor of safety*/
      agx::Real             fos_n = 0.0;

      ~SubStepResult() = default;
    };

    using SubStepResultVector = agx::Vector<SubStepResult>;

    /*
    Results from the ground collapse algorithm for a specific FailureSurface.
    */
    struct AGXTERRAIN_EXPORT FailureSurfaceResult
    {
      /*
      The FailureSurface.
      */
      FailureSurfaceRef failureSurface = nullptr;

      /*
      SoilAggregate representation of FailureSurface.
      */
      FailureSurfaceAggregateRef soilAggregate = nullptr;

      /*
      Whether the failure surface has been tested for ground collapse.
      */
      bool solved = false;

      /*
      Whether the failure surface collapsed according to the given failure criteria.
      */
      bool failed = false;

      /*
      Load exerted on the soil aggregate during simulation.
      */
      LoadPointVector loadPoints;

      /*
      Critical factor of safety
      */
      agx::Real fos = 0.0;

      /*
      Critical weighted factor of safety
      */
      agx::Real fos_n = 0.0;

      /*
      Linear acceleration
      */
      agx::Vec3 la = { 0,0,0 };

      /*
      Angular acceleration
      */
      agx::Vec3 aa = { 0,0,0 };

      /*
      Results from each substep solve in the ground collapse simulation.
      */
      SubStepResultVector solveResults;
    };
    using FailureSurfaceResultVector = agx::Vector<FailureSurfaceResult>;


   /*
   Results from the ground collapse simulation.

   Contains results from all the failure surfaces simulated.
   */
    AGX_DECLARE_POINTER_TYPES( GroundCollapseResult );
    class AGXTERRAIN_EXPORT GroundCollapseResult : public agx::Referenced
    {
    public:
      /**
      * Default constructor.
      */
      GroundCollapseResult();

      /**
      *\return number of solved failure surfaces.
      */
      agx::UInt getNumSolved() const;

      /**
      *\return number of failure surfaces that have failed, i.e where we have soil collapse.
      */
      agx::UInt getNumFailed() const;

      /**
      *\return the total sum of voxels in all failure surfaces
               processed by the ground collapse calculations.
      */
      agx::UInt getTotalNumSurfaceVoxels() const;

      /**
      *\return the critical factor of saftey for all processed failure surfaces.
      */
      agx::Real getCriticalFos() const;

      /**
      *\return the index to the failure surface in the result vector that
               contains the critical factor of saftey.
               note: get the result vector from the getResults() function.
      */
      int getCriticalFosIndex() const;

      /**
      \return a vector of FailureSurfaceResult that contains the results for all
              processed failure surfaces in the ground collapse calculations.
      */
      FailureSurfaceResultVector& getResults();

      DOXYGEN_START_INTERNAL_BLOCK()
      void reserve( size_t size );
      void addResult( const FailureSurfaceResult& result );
      DOXYGEN_END_INTERNAL_BLOCK()

    protected:
      virtual ~GroundCollapseResult() {};

    protected:
      agx::UInt m_numSolved;
      agx::UInt m_numFailed;
      agx::UInt m_totSurfaceVoxels;
      int m_criticalFosIndex;
      agx::Real m_criticalFos;
      FailureSurfaceResultVector m_results;
    };
  }
}