/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceOutput.h>
#include <agxSensor/RaytraceRemoveRayMisses.h>
#include <agxSensor/RaytraceOutputNoise.h>
#include <agxSensor/SensorOutputHandler.h>

#include <agx/HashTable.h>

namespace agxSensor
{
  class RtNodeGroupHandler;

  AGX_DECLARE_POINTER_TYPES( RtOutputHandler );

  /**
  Raytrace result handler is resposible for when and how the
  raytracing starts and ends and collecting/fetching data.
  */
  class AGXSENSOR_EXPORT RtOutputHandler : public RtSystemNode, public ISensorOutputHandler
  {
    public:
      enum RuntimeFlag : agx::UInt32
      {
        RAYTRACING = 1 << 0,
        FETCHING_DATA = 1 << 1,
        RESTORED = 1 << 2
      };

      using RuntimeState = agx::AtomicState<RuntimeFlag>;

    public:
      /**
      Default constructor.
      */
      RtOutputHandler();

      /**
      Add raytrace result given unique id, e.g., RtOutput::getTypeHash<MyData>().
      The resulting data of a raytrace is accessed through the given \p result
      instance.
      \param uniqueId - unique id > 0, id 0 is reserved and invalid to use
      \param output - output instance to add
      \return true if \p uniqueId > 0, \p result != nullptr and \p uniqueId is unique
      */
      bool add( size_t uniqueId, RtOutput* output );

      /**
      Build matching result to a given data type T. This method verifies the
      size of the sum of data fields matches the size of T. The created RtOutput
      returned from this method can later be accessed using get<T>().
      \return created instance if successful (matching size), otherwise nullptr
      */
      template<typename T, RtOutput::Field... field>
      RtOutput* add();

      /**
      Access result instance given the unique id used when it was added.
      \param uniqueId - unique id for the result instance
      \return result instance for the given unique id if it exists, otherwise nullptr
      */
      RtOutput* get( size_t uniqueId );

      /**
      Access result instance given the unique id used when it was added.
      \param uniqueId - unique id for the result instance
      \return result instance for the given unique id if it exists, otherwise nullptr
      */
      ISensorOutput* getBase( size_t uniqueId ) override;

      /**
      Access added result instance for the given data type T. E.g.,
      handler->add<MyData, ...>() == handler->get<MyData>().
      \return the result instance for the given type T if it exists, otherwise nullptr
      */
      template<typename T>
      RtOutput* get();

      /**
      Directly access an output view of the raytrace output instance using a hash of the type with
      which it was registered.
      \return output view of the raytrace output if an output with the given type exists and if the
              view type aligns with the data, otherwise an empty output view is returned
      */
      template<typename T>
      BinaryOutputView<T> view();

      /**
      Configure the number of maximum raytrace steps.
      The number of steps is one more than the number of bounces a ray will make, 
      however, this number will generally not affect the size of the output data.
      It should be noted that the time and memory complexity of the raytrace will grow
      exponentially with the maximum number of raytrace steps.
      \param steps - maximum number of steps a ray will trace.
      */
      void setRaytraceDepth( size_t steps = 1 );

      /**
      \return current number of maximum raytrace steps.
      */
      size_t getRaytraceDepth() const;

      /**
      Set the seed used to configure the pseudo-random generation the ambient/atmospheric hits.
      \param seed - value to seed the pseudo-random generator with
      */
      void setAmbientHitSeed( agx::UInt32 seed = 783u );

      /**
      \return the seed used to configure the pseudo-random generation the ambient/atmospheric hits.
      */
      agx::UInt32 getAmbientHitSeed() const;

      /**
      Enable/disable pruning of data for rays that doesn't hit any target.
      This is enabled by default to minimize the output data.
      \param enable - true to enable pruning of data from ray that doesn't hit,
                      false to have data one to one with the number of rays but
                      RtOutput::IS_HIT_I32 or similar has to be checked instead.
      */
      void setEnableRemoveRayMisses( bool enable );

      /**
      \return true if data from rays that misses any target is removed, false and
              the output data is one to one with the number of rays from the lidar.
      */
      bool getEnableRemoveRayMisses() const;

      /**
      Add output noise instance to the system tree.
      \param noise - output noise to add to the system tree
      \return true if added successfully, false if nullptr or already added
      */
      bool add( RtOutputNoise* noise );

      /**
      Remove output noise instance from the system tree. The noise will no longer
      be applied to the output data if removed successfully.
      \param noise - noise to remove
      \return true if removed successfully, false if nullptr or not part of this
              system tree
      */
      bool remove( RtOutputNoise* noise );

      /**
      \param noise - noise instance to check if present and active
      \return true if \p noise is part of this output handler
      */
      bool contains( const RtOutputNoise* noise ) const;

      /**
      \return all added and active output noises
      */
      RtOutputNoiseRefVector getOutputNoises() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      using ISensorOutputHandler::view;

    public:

      /**
      \return system node proxy for the given noise instance
      */
      RtSystemNodeProxy* getProxy( const RtOutputNoise* noise ) const;

      /**
      Initializes the final structure with the raytrace node as child to
      the given parent.
      */
      virtual void onParentConnect( RtSystemNode& parent ) override;

      /**
      Start the raytrace simulation, non-blocking.
      */
      virtual void execute( const CallbackData& data ) override;

      /**
      Complete the raytrace simulation, blocking.
      */
      virtual void complete() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtOutputHandler );

      /**
      Inserts new parent to this node. The previous parent to this node
      will get \p parent as child.
      \param parent - new parent to this node
      \return true if successful, false if initialization of RtNode of \p parent failed
      */
      virtual bool insertParent( SystemNode& parent ) override;

    protected:
      /**
      \return the created raytrace node
      */
      virtual RtNode* createNode() final override;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      struct OutputNoiseData
      {
        RtOutputNoiseRef noise;
        RtSystemNodeRef proxy;
      };

      bool tryAssembleLocalRtOutputChain();

    private:
      using OutputTable = agx::HashTable<size_t, RtOutputRef>;
      using OutputNoiseDataVector = std::vector<OutputNoiseData>;

    private:
      RtNodeGroupHandler* getExitNode() const;

    private:
      OutputTable m_outputs;
      RuntimeState m_runtimeState;
      RtNode* m_parentNode;
      size_t m_raytraceDepth;
      RtNode* m_raytraceNode;
      RtNode* m_beamDivergenceNode;
      agx::UInt32 m_ambientHitSeed;
      RtNode* m_assemblyNode;
      RtRemoveRayMissesRef m_removeRayMisses;
      OutputNoiseDataVector m_outputNoises;
  };

  template<typename T, RtOutput::Field... field>
  RtOutput* RtOutputHandler::add()
  {
    RtOutputRef output = new RtOutput();
    output->build<field...>();

    if ( !RtOutput::template verifySize<T>( output->getElementSize() ) )
      return nullptr;

    if ( !this->add( RtOutput::template getTypeHash<T>(), output ) )
      return nullptr;

    return output;
  }

  template<typename T>
  RtOutput* RtOutputHandler::get()
  {
    return this->get( RtOutput::template getTypeHash<T>() );
  }

  template<typename T>
  BinaryOutputView<T> RtOutputHandler::view()
  {
    return this->view<T>( RtOutput::template getTypeHash<T>() );
  }
}
