/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <agxSensor/RaytraceSystemNodeProxy.h>

#include <agx/agx_vector_types.h>

#include <agxStream/OutputArchive.h>
#include <agxStream/InputArchive.h>

#include <memory>

#ifdef _MSC_VER
#pragma warning(push)
#pragma warning(disable: 4251) // Disable warnings about members needing dll interface.
#endif

namespace agxSensor
{
  struct LidarRayPatternStorage;
  struct LidarRayPatternInterval;
  struct RtNode;

  AGX_DECLARE_POINTER_TYPES(LidarRayPatternGenerator);

  /**
  Base of lidar ray pattern implementations resposible of providing a set
  of ray transforms (ray along z) and an interval each tick indexing into
  the set of ray transforms.
  */
  class AGXSENSOR_EXPORT LidarRayPatternGenerator : public IRtSystemNode
  {
    public:
      /**
      Default constructor.
      */
      LidarRayPatternGenerator();

      /**
      Assign all, locally possible, ray transforms. The range of active rays
      each step is given by getNextInterval. A ray defined to be along the z
      axis of the transform and the transform is local to the lidar.
      \param rayTransforms - transforms of the rays, the range returned from getNextInterval
                             is the ones that are used/visible
      */
      void setRayTransforms(const agx::AffineMatrix4x4Vector& rayTransforms);

      /**
      Apply delta transform to all current ray transforms.
      \param rhs - delta transform
      */
      void applyTransform(const agx::AffineMatrix4x4& rhs);

    public:
      /**
      Destructor.
      */
      virtual ~LidarRayPatternGenerator() = default;

      /**
      Provides the active ray index range for a given step length \p dt. The
      returned index range is used on the transforms passed to setRayTransforms.
      \param dt - step size in seconds
      \return index range at the current time spanning the given step size \p dt
      */
      virtual LidarRayPatternInterval getNextInterval(agx::Real dt) = 0;

      /**
      \return the maximum number of rays this pattern can generate
      */
      virtual size_t getNumRays() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual RtNode* createNode() override;

      virtual void synchronize(RtNode* node, const CallbackData& data) override;

      virtual void store( agxStream::OutputArchive& out ) const override;
      virtual void restore( agxStream::InputArchive& in ) override;

      void store( agxStream::OutputArchive& out, bool storeTransforms ) const;
      void restore( agxStream::InputArchive& in, bool restoreTransforms );

    protected:
      template<typename ContainerT>
      static void writeTransforms( agxStream::OutputArchive& out, const ContainerT& container );

      template<typename ContainerT>
      static void readTransforms( agxStream::InputArchive& in, ContainerT& container );

    protected:
      LidarRayPatternStorage* getStorage();
      const LidarRayPatternStorage* getStorage() const;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      void updateRayTransforms(RtNode* node, agx::Real dt);

    private:
      std::shared_ptr<LidarRayPatternStorage> m_storage;
  };

  template<typename ContainerT>
  void LidarRayPatternGenerator::writeTransforms( agxStream::OutputArchive& out,
                                                  const ContainerT& container )
  {
    const size_t numRayTransforms = container.size();
    out << agxStream::out( "numRayTransforms", numRayTransforms );

    out.beginSection( "rayTransforms" );
    out.write( container.data(), sizeof( typename ContainerT::value_type ) * numRayTransforms );
    out.endSection( "rayTransforms" );
  }

  template<typename ContainerT>
  void LidarRayPatternGenerator::readTransforms( agxStream::InputArchive& in,
                                                 ContainerT& container )
  {
    size_t numRayTransforms = 0u;
    in >> agxStream::in( "numRayTransforms", numRayTransforms );

    container.resize( numRayTransforms );
    in.beginSection( "rayTransforms" );
    in.read( container.data(), sizeof( typename ContainerT::value_type ) * numRayTransforms );
    in.endSection( "rayTransforms" );
  }


  /**
  Ray pattern interval (index range) with a start index and size.
  */
  struct AGXSENSOR_EXPORT LidarRayPatternInterval
  {
    /**
    Default interval of zero size and start index 0.
    */
    LidarRayPatternInterval() = default;

    /**
    Construct given start index \p first and number of size \p numRays
    from \p first. [first, first + numRays)
    \param first - first index
    \param numRays - number of rays (size of the interval)
    */
    LidarRayPatternInterval(agx::Int32 first, agx::Int32 numRays)
      : first(first)
      , numRays(numRays)
    {
    }

    agx::Int32 first{0};
    agx::Int32 numRays{0};
  };
}

#ifdef _MSC_VER
#pragma warning(pop)
#endif
