/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <memory>
#include <unordered_map>
#include <agx/ref_ptr.h>
#include <agx/Referenced.h>
#include <agx/ElementaryConstraint.h>
#include <agx/Logger.h>
#include <agxPowerLine/Connector.h>
#include <agxSDK/Assembly.h>
#include <agxPowerLine/PowerLine.h>
#include <agxOpenPLX/export.h>
#include <openplx/UuidGenerator.h>

namespace openplx::Core {
    class Object;
}

namespace agxopenplx {

    enum class SignalSourceMapMode {
        Name,
        Uuid
    };

    /**
     * Used to map OpenPLX instances to AGX instances either using the OpenPLX
     * instance UUID remapped with the OpenPLX instance type using UUIDv5,
     * or the OpenPLX instance name.
     */
    class AGXOPENPLX_EXPORT SignalSourceMapper
    {
        public:

            /**
             * Creates a signal source mapper that looks up AGX instances from an assembly and a power line.
             * \param assembly - The assembly used to lookup AGX instances from OpenPLX instances
             * \param power_line - The power line used to lookup AGX instances from OpenPLX instances
             * \param map_mode - The method to map OpenPLX instances to AGX instances (by Uuid by default).
             */
            static std::shared_ptr<SignalSourceMapper> create(
                agxSDK::Assembly* assembly,
                agxPowerLine::PowerLine* power_line = nullptr,
                SignalSourceMapMode map_mode = SignalSourceMapMode::Uuid
            );

            /**
             * Creates a signal source mapper that looks up AGX instances from pre filled map.
             * \param ref_map - The pre filled map mapping keys to AGX instances
             * \param map_mode - The method to map OpenPLX instances to keys for the ref_map (Uuid by default).
             */
            static std::shared_ptr<SignalSourceMapper> createPreMapped(
                std::unordered_map<std::string, agx::Referenced *> ref_map,
                SignalSourceMapMode map_mode = SignalSourceMapMode::Uuid
            );

            /**
             * Finds an AGX instances from an OpenPLX instance
             * \param object_or_trait - The OpenPLX instance
             */
            template<class T, class S>
            agx::ref_ptr<T> findMapped(
                const std::shared_ptr<S>& object_or_trait) {
                static_assert(std::is_base_of<agx::Referenced, T>::value, "T must derive from agx::Referenced");
                auto object = std::dynamic_pointer_cast<openplx::Core::Object>(object_or_trait);
                if (object == nullptr) return nullptr;
                auto raw = findMappedRaw(object);
                if (raw != nullptr) {
                    auto specific = dynamic_cast<T*>(raw);
                    if (specific != nullptr) {
                        return agx::ref_ptr<T>(specific);
                    }
                } else if (m_assembly != nullptr) {
                    if constexpr (std::is_same_v<T, agx::ObserverFrame>) {
                        if (m_map_mode == SignalSourceMapMode::Uuid && object->getType() != nullptr) {
                            auto uuid = agx::Uuid(
                                openplx::UuidGenerator::generateUuidv5(
                                    object->getUuid(), object->getType()->getNameWithNamespace(".")));
                            return m_assembly->getObserverFrame(uuid);
                        } else if (m_map_mode == SignalSourceMapMode::Name) {
                            return m_assembly->getObserverFrame(object->getName());
                        }
                    } else if constexpr (std::is_same_v<T, agx::RigidBody>) {
                        if (m_map_mode == SignalSourceMapMode::Uuid && object->getType() != nullptr) {
                            auto uuid = agx::Uuid(
                                openplx::UuidGenerator::generateUuidv5(
                                    object->getUuid(), object->getType()->getNameWithNamespace(".")));
                            return m_assembly->getRigidBody(uuid);
                        } else if (m_map_mode == SignalSourceMapMode::Name) {
                            return m_assembly->getRigidBody(object->getName());
                        }
                    } else if constexpr (std::is_base_of_v<agx::Constraint, T>) {
                        if (m_map_mode == SignalSourceMapMode::Uuid && object->getType() != nullptr) {
                            auto uuid = agx::Uuid(
                                openplx::UuidGenerator::generateUuidv5(
                                    object->getUuid(), object->getType()->getNameWithNamespace(".")));
                            agx::ConstraintRef constraint = m_assembly->getConstraint(uuid);
                            return agx::dynamic_pointer_cast<T>(constraint);
                        } else if (m_map_mode == SignalSourceMapMode::Name) {
                            agx::ConstraintRef constraint = m_assembly->getConstraint(object->getName());
                            return agx::dynamic_pointer_cast<T>(constraint);
                        }
                    } else if constexpr (std::is_base_of_v<agxSDK::Assembly, T>) {
                        if (m_map_mode == SignalSourceMapMode::Uuid && object->getType() != nullptr) {
                            auto uuid = agx::Uuid(
                                openplx::UuidGenerator::generateUuidv5(
                                    object->getUuid(), object->getType()->getNameWithNamespace(".")));
                            agxSDK::AssemblyRef assembly = m_assembly->getAssembly(uuid);
                            return agx::dynamic_pointer_cast<T>(assembly);
                        } else if (m_map_mode == SignalSourceMapMode::Name) {
                            agxSDK::AssemblyRef assembly = m_assembly->getAssembly(object->getName());
                            return agx::dynamic_pointer_cast<T>(assembly);
                        }
                    }
                }
                LOGGER_WARNING() << "Unable to map " << object->getName() << " to " << typeid(T).name() << LOGGER_ENDL();
                return nullptr;
            }

DOXYGEN_START_INTERNAL_BLOCK()
            agx::Referenced* findMappedRaw(
                const std::shared_ptr<openplx::Core::Object>& object);

            static void populateWithConstraints(
                agxSDK::Assembly& assembly,
                std::unordered_map<std::string, agx::Referenced*>& ref_map,
                SignalSourceMapMode map_mode
            );

            static void populateWithSecondaryConstraints(
                agxSDK::Assembly& assembly,
                std::unordered_map<std::string, agx::Referenced*>& ref_map,
                SignalSourceMapMode map_mode
            );

            static void populateWithUnits(
                agxPowerLine::PowerLine& power_line,
                std::unordered_map<std::string, agx::Referenced*>& ref_map,
                SignalSourceMapMode map_mode
            );

            static void populateWithConnectors(
                agxPowerLine::PowerLine& power_line,
                std::unordered_map<std::string, agx::Referenced*>& ref_map,
                SignalSourceMapMode map_mode
            );
DOXYGEN_END_INTERNAL_BLOCK()

        private:
            SignalSourceMapper(
                SignalSourceMapMode map_mode
            );
            agxSDK::AssemblyRef m_assembly;
            std::unordered_map<std::string, agx::Referenced*> m_ref_map;
            SignalSourceMapMode m_map_mode;
    };
}
