/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Referenced.h>
#include <agx/Vector.h>
#include <agx/agx_vector_types.h>
#include <agx/Constraint.h>
#include <agx/RigidBody.h>
#include <agx/AffineMatrix4x4.h>

#include <agxModel/export.h>


namespace agxSDK {
  class Simulation;
}

namespace agxModel
{

  /**
  A KinematicChain represents a chain consisting of multiple agx::RigidBody
  and agx::Constraint objects.

  \note
  Depending on the field of application, the rigid bodies can be referred to as links or segments.
  The constraints are commonly called joints.

  */
  class AGXMODEL_EXPORT KinematicChain : public agx::Referenced
  {
    public:

      enum KinematicChainStatus {
        SUCCESS,               ///< Return value indicating success.
        INVALID_CHAIN,         ///< Failure due to chain not being valid
        BAD_INPUT_ARGUMENTS,   ///< Failure due to input arguments being incorrect. E.g. wrong number of vector elements.
        SOLVER_FAILURE,        ///< Failure due to solver not finding a solution. E.g. pose not reachable.
        NO_SIMULATION,         ///< Failure due to operation requiring access to the Simulation holding bodies/constraints.
      };

  };



  /**
  A Serial Kinematic Chain is a KinematicChain where there are no loops or branches.

  When a chain is specified, it is via a series of agx::RigidBody and agx::Constraint objects where
  the number of rigid bodies is one more than number of constraints.

  Bodies are expected to be handled as 6-dof bodies and having setHandleAsParticle enabled
  is not supported.

  The constraint types that are supported are:
  - Rotational / Hinge
  - Sliding / Prismatic
  - Fixed / LockJoint

  The first two constraint types remove 5 degrees of freedom (dof) each where as the lock
  removes 6 dofs.

  After the chain representation is created, querying the number of joints in the chain will give
  the number of constraints that leave 1 degree of freedom (rotation or sliding) for the connected
  body. This is the same number of joint values which should be provided to the
  computeForwardKinematics function or are outputted from computeInverseKinematics or
  computeInverseDynamics.

  If the coordinate frame of interest does not align with the model coordinate system for the
  final body/link/segment in the chain, then there is a possibility to append a local transform
  in the end links coordinate system. The appended transform will be the final transform from the
  chain. This optional appended transform is fixed relative the last body and does not increase
  the number of joint values that are expected as input or produced as output. It does increase
  the number of link outputs with one.

  \note
  Kinematic computations can be performed if the chain is valid. Performing inverse dynamics
  computations has one additional requirement: being able to access data in the simulation
  that contains the bodies and constraints.
  If a constructor is used that does not take a Simulation as parameter, then calling setSimulation
  must be performed before computeInverseDynamics can be used.
  */
  class AGXMODEL_EXPORT SerialKinematicChain : public KinematicChain
  {
    public:

      /**
      Create a serial chain from "base" to "end" by traversing the bodies and constraints in the
      simulation.

      The chain will contain relative transforms from when it was created. Any changes to
      e.g. a constraint attachment frame will not be seen. If the kinematic structure is changed,
      then the chain must be recreated.

      \param simulation The simulation in which to traverse bodies and constraints to find a
             chain between "base" and "end".
      \param base The first body in the chain
      \param end The last body in the chain
      \param endLocalTransform An optional transform expressed in the last bodies frame which
             extends the chain with one additional link.
      */
      SerialKinematicChain(agxSDK::Simulation& simulation,
                           const agx::RigidBody& base,
                           const agx::RigidBody& end,
                           const agx::AffineMatrix4x4* endLocalTransform = nullptr);


      /**
      Create a serial chain from the input bodies and constraints which are provided in the
      correct order.

      The chain will contain relative transforms from when it was created. Any changes to
      e.g. a constraint attachment frame will not be seen. If the kinematic structure is changed,
      then the chain must be recreated.

      \note
      Number of bodies should be equal to number of constraints + 1.
      The first body in the bodies vector is "base" and the last one is "end".
      For well formed input, bodies[i] and bodies[i+1] are connected via constraints[i].

      \param bodies A vector with the bodies in the chain
      \param constraints A vector with constraints that connect the bodies
      \param endLocalTransform An optional transform expressed in the last bodies frame
             which extends the chain with one additional link.
      */
      SerialKinematicChain(const agx::RigidBodyPtrVector& bodies,
                           const agx::ConstraintPtrVector& constraints,
                           const agx::AffineMatrix4x4* endLocalTransform = nullptr);



      /**
      Set the Simulation that is used to read certain values needed
      for inverse dynamics computations. Calling this method is needed
      if a constructor was used that did not take a Simulation as argument.
      */
      void setSimulation(agxSDK::Simulation* sim);


      /**
      Check if the chain was valid at time of creation.

      Modifying bodies (e.g. enabling treat as particle) or
      constraints (e.g. adding additional elementary constraints) will
      in many cases cause compute methods to return INVALID_CHAIN regardless
      of if this method returns true or not.

      \return True if a valid was constructed.
      */
      bool isValid() const;


      /**
      Returns number of links in the chain.

      This number often corresponds to the number of bodies that were used
      when creating the chain representation.
      If a final "end transform" was specified, then the link count is increased by 1.
      */
      agx::UInt getNumLinks() const;


      /**
      Returns number of joints in the chain where motion can occur.

      This number does not have to match the input number of constraints since it only includes
      joints which have 1 degree of freedom (rotation or sliding).

      Inputs and outputs in joint space should have this number of elements.
      */
      agx::UInt getNumJoints() const;


      /**
      Get the links that are in the chain.
      */
      const agx::RigidBodyRefVector& getLinks() const;


      /**
      Get the joints that are in the chain.
      */
      const agx::ConstraintRefVector& getJoints() const;


      /**
      Compute the transform for the end of the chain relative the base given the provided
      joint values.

      By default, the pose is computed given the provided values as is regardless if the joints
      have limits or not.

      \param jointPositions Vector with the same number of joint values as there are joints
             in the chain.
      \param outputTransform The transform for the end body relative the base. Only written if
             method returns success.
      \param clampToJointLimits If true, then each value in jointPositions will be clamped to
             corresponding joints range before computing the pose.
      \returns Status code indicating if function succeeded.
      */
      KinematicChainStatus computeForwardKinematics(const agx::RealVector& jointPositions,
                                                    agx::AffineMatrix4x4& outputTransform,
                                                    bool clampToJointLimits = false);


      /**
      Compute the transform for all links in the chain relative the base given provided
      joint values. For a chain with N links, there will be N AffineMatrix4x4 in the output vector.
      The first item for "base" will be an identity matrix.

      By default, the poses are computed given the provided values as is regardless if the
      joints have limits or not.

      \param jointPositions Vector with the same number of joint positions as there are
             joints in the chain.
      \param outputTransforms Vector with transforms. Only written if method returns success.
      \param clampToJointLimits If true, then each value in jointPositions will be clamped to
              corresponding joints range before computing the pose.
      \returns Status code indicating if function succeeded.
      */
      KinematicChainStatus computeForwardKinematicsAll(const agx::RealVector& jointPositions,
                                                       agx::AffineMatrix4x4Vector& outputTransforms,
                                                       bool clampToJointLimits = false);


      /**
      Compute the transforms and velocities for all links in the chain relative the base
      given provided joint values.

      By default, the poses are computed given the provided values as is regardless if the
      joints have limits or not.

      \param jointPositions Vector with the same number of joint positions as there are
             joints in the chain.
      \param jointVelocities Vector with the same number of joint velocities as there
             are joints in the chain.
      \param outputTransforms Vector with transforms. Only written if method returns success.
      \param outputLinearVelocities Vector with computed linear velocity for each link.
      \param outputAngularVelocities Vector with computed angular velocity for each link.
      \param clampToJointLimits If true, then each value in jointPositions will be clamped to
             corresponding joints range before computing the pose.
      \returns Status code indicating if function succeeded.
      */
      KinematicChainStatus computeForwardKinematicsAll(const agx::RealVector& jointPositions,
                                                       const agx::RealVector& jointVelocities,
                                                       agx::AffineMatrix4x4Vector& outputTransforms,
                                                       agx::Vec3Vector& outputLinearVelocities,
                                                       agx::Vec3Vector& outputAngularVelocities,
                                                       bool clampToJointLimits = false);


      /**
      Compute joint positions needed for the end body in the chain to get the specified transform
      relative the base.

      \param transform Desired transform for the end body relative the base.
      \param currentJointPositions Vector with current positions for each joint
             which has a degree of freedom.
      \param outputJointPositions Vector to which joint values will be stored.
      \param solverEpsilon precision for IK solution
      \returns Status code indicating if function succeeded.
      */
      KinematicChainStatus computeInverseKinematics(const agx::AffineMatrix4x4& transform,
                                                    const agx::RealVector& currentJointPositions,
                                                    agx::RealVector& outputJointPositions,
                                                    agx::Real solverEpsilon=1e-5);


      /**
      Compute the torque/forces needed to get the specified jointAccelerations. Current joint
      positions and velocities are read from bodies and constraints forming the chain.

      \param jointAcceleration Desired joint acceleration.
      \param tau Output torques/forces.
      \return Status code indicating if function succeeded.
      */
      KinematicChainStatus computeInverseDynamics(const agx::RealVector& jointAcceleration,
                                                  agx::RealVector& tau);


      /**
      \copydoc SerialKinematicChain::computeInverseDynamics

      \param externalForces Extra external forces acting on the bodies. This vector should have
                            6 elements (3 force, 3 torque) for each body in the chain.
      \return Status code indicating if function succeeded.
      */
      KinematicChainStatus computeInverseDynamics(const agx::RealVector& jointAcceleration,
                                                  const agx::RealVector& externalForces,
                                                  agx::RealVector& tau);


      /**
      Compute the manipulator jacobian relative the base. This jacobian gives a relationship
      between end effector velocities and joint velocities.

      \param jointPositions Joint positions for which the jacobian should be evaluated
      \param jacobian A vector where the output 6xN jacobian is stored by row.
      \return Status code indicating if function succeeded.
      */
      KinematicChainStatus computeManipulatorJacobian(const agx::RealVector& jointPositions,
                                                      agx::RealVector& jacobian);
    protected:
      class SerialKinematicChainData;

      SerialKinematicChain();
      ~SerialKinematicChain();

    private:
      using SerialKinematicChainDataRef = agx::ref_ptr< SerialKinematicChainData >;

      SerialKinematicChainDataRef m_data;
  };

  AGX_DECLARE_POINTER_TYPES(KinematicChain);
  AGX_DECLARE_POINTER_TYPES(SerialKinematicChain);

}
