/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Constraint.h>

namespace agx
{
  class GenericConstraintComplianceMatrix;
  class GenericConstraintImplementation;

  class AGXPHYSICS_EXPORT IGenericConstraint : public Constraint
  {
    public:
#ifndef SWIG
      static void onPreSolveCallback( const agxSDK::Simulation* simulation );
      static void onPostSolveCallback( const agxSDK::Simulation* simulation );
      static void onCleanup( const agxSDK::Simulation* simulation );
#endif

    public:
      /**
      Enable/disable updateComplianceMatrix callback with write access
      to the elements in the compliance matrix of this constraint. If
      \p isStaticSize is true, meaning the number of active rows in this
      constraint is constant, the matrix is accessible and wont reset
      to zero each time step.
      \note This is only supported by the direct solver.
      \param enable - true to enable, false to disable
      \param isStaticSize - true if this constraints active rows is constant, e.g., LockJoint,
                            enabling access to the compliance matrix in runtime and the
                            matrix isn't reset to zero each time step.
      */
      void setSupportComplianceMatrix( Bool enable, Bool isStaticSize = false );

      /**
      If this constraint active rows doesn't vary and isStaticSize = true has
      been flagged when enabling support for compliance matrix, this method
      will return the matrix at any time. Otherwise nullptr is returned.
      \return compliance matrix if accessible, otherwise nullptr
      */
      GenericConstraintComplianceMatrix* getComplianceMatrix() const;

      /**
      Assign/update rigid bodies of this constraint. Note that \p rb1 must be
      valid for this constraint to be valid.
      \param rb1 - first rigid body
      \param rb2 - second rigid body
      \return true if this constraint is valid after the bodies has been assigned, otherwise false
      */
      Bool setRigidBodies( RigidBody* rb1, RigidBody* rb2 );

      /**
      Add child constraint (binary or many-body).
      \note It's up to the user to make sure this child constraint
      isn't added multiple times.
      \param constraint - child constraint to add
      \return true if added, false if nullptr.
      */
      Bool addChild( Constraint* constraint );

      /**
      Remove child constraint.
      \param constraint - child constraint to remove
      \return true if removed, otherwise false.
      */
      Bool removeChild( Constraint* constraint );

      /**
      Removes all child constraints previously added.
      */
      void removeAllChildren();

      /**
      Register rigid body part of this many-body constraint.
      \param rb - rigid body to add
      \return true if added, false if nullptr.
      */
      Bool add( RigidBody* rb );

      /**
      Remove rigid body from this many-body constraint.
      \param rb - rigid body to remove
      \return true if removed, otherwise false.
      */
      Bool remove( RigidBody* rb );

      /**
      Removes all bodies previously added.
      */
      void removeAllBodies();

    public:
      virtual ~IGenericConstraint();
      virtual void addNotification() final override;
      virtual void removeNotification() final override;

      virtual void onAddNotification() = 0;
      virtual void onRemoveNotification() = 0;

      virtual bool onPreSolve() = 0;
      virtual void onPostSolve() = 0;

      virtual void updateComplianceMatrix( GenericConstraintComplianceMatrix* matrix ) = 0;

    public:
      int getNumDOF() const override;
      void render( class agxRender::RenderManager* manager, float scale ) const override;

    protected:
      IGenericConstraint( GenericConstraintImplementation* implementation );

    private:
      ConstraintRefVector m_children;
  };
}
