/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/ConstraintImplementation.h>

namespace agx
{
  /**
  Constraint body data used with GenericElementaryConstraint containing
  the per body Jacobian data.
  */
  class AGXPHYSICS_EXPORT GenericConstraintBodyData
  {
    public:
      /**
      Construct given number of rows in the elementary constraint.
      */
      GenericConstraintBodyData();

      /**
      Assign Jacobian for given row as spatial and rotation vectors.
      \param row - row in the Jacobian
      \param spatial - spatial part of the Jacobian
      \param rotational - rotational part of the Jacobain (ignored if particle)
      */
      void setJacobian( UInt row, const Vec3& spatial, const Vec3& rotational );

      /**
      Add Jacobian for given row as spatial and rotation vectors.
      \param row - row in the Jacobian
      \param spatial - spatial part of the Jacobian
      \param rotational - rotational part of the Jacobain (ignored if particle)
      */
      void addJacobian( UInt row, const Vec3& spatial, const Vec3& rotational );

      /**
      \return the Jacobian data where index 2 * i is spatial part and 2 * i + 1 the rotational
      */
      const Vec3Vector& getJacobian() const;

      /**
      Resets all vectors to (0, 0, 0).
      */
      void reset( UInt numRows );

    private:
      Vec3Vector m_jacobian;
  };

  /**
  Constraint data used with GenericElementaryConstraint containing
  data such as; constraint violation, constraint velocity and bounds.
  */
  class AGXPHYSICS_EXPORT GenericConstraintData
  {
    public:
      using BoundsVector = VectorPOD<RangeReal>;

    public:
      /**
      Construct given number of rows in the elementary constraint.
      */
      GenericConstraintData();

      /**
      Assign violation for given row in the elementary constraint.
      \param row - elementary constraint row
      \param violation - elementary constraint violation
      */
      void setViolation( UInt row, Real violation );

      /**
      \return the violations
      */
      const RealVector& getViolation() const;

      /**
      Assign velocity for given row in the elementary constraint.
      \param row - elementary constraint row
      \param velocity - elementary constraint velocity
      */
      void setVelocity( UInt row, Real velocity );

      /**
      \return the velocities
      */
      const RealVector& getVelocity() const;

      /**
      Assign bound for given row in the elementary constraint.
      \note The bound given by the user has already been assigned
            when the GenericElementaryConstraint receives its callback.
      \param row - elementary constraint row
      \param range - elementary constraint bound
      */
      void setBound( UInt row, RangeReal range );

      /**
      \return the bounds
      */
      const BoundsVector& getBounds() const;

      /**
      Assign impacting state of the elementary constraint.
      Default: false.
      \param isImpacting - true if the elementary constraint is impacting, otherwise false
      */
      void setIsImpacting( Bool isImpacting );

      /**
      \return true if the elementary constraint is impacting, otherwise false
      */
      Bool getIsImpacting() const;

      /**
      Resets all values to default.
      */
      void prepare( UInt numRows, const ConstraintRigidBodyContainer& bodies );

      /**
      Access body related data, such as the Jacobian, for a given body \p rb.
      \param rb - rigid body to access data for, rb == nullptr will return valid
                  but the data is ignored
      \return body data (such as the Jacobian) for the given instance \p rb
      */
      GenericConstraintBodyData* getBodyData( const RigidBody* rb ) const;

      /**
      Access body related data, such as the Jacobian, for a given body index \p index.
      For an ordinary binary constraint with a transformed attachment pair, it's then
      possible to do:

          a1: agx.Attachment = self.getAttachmentPair().getAttachment1()
          a2: agx.Attachment = self.getAttachmentPair().getAttachment2()
          rb1_data: agx.GenericConstraintBodyData = constraintData.getBodyDataAt(0)
          rb2_data: agx.GenericConstraintBodyData = constraintData.getBodyDataAt(1)
          v1 = a1.get(agx.Attachment.N)
          v2 = a2.get(agx.Attachment.N)
          rb1_data.setJacobian(0, agx.Vec3(), v1)
          rb2_data.setJacobian(0, agx.Vec3(), -v2)

      \param index - rigid body index to access data for
      \return body data (such as the Jacobian) for the given index \p index
      */
      GenericConstraintBodyData* getBodyDataAt( size_t index ) const;

    private:
      using BodyDataContainer = Vector<GenericConstraintBodyData>;

    private:
      const ConstraintRigidBodyContainer* m_bodies;
      RealVector m_violation;
      RealVector m_velocity;
      BoundsVector m_bounds;
      Bool m_isImpacting;
      BodyDataContainer m_bodyData;
  };
}
