#pragma once

#include <memory>
#include <string>
#include <optional>
#include <unordered_map>
#include <cstdint>
#include <openplx/core_export.h>

namespace openplx {

    namespace MarshallingTemplateUtils {
        template <class T>
        using remove_cvref_t = std::remove_cv_t<std::remove_reference_t<T>>;

        // true only for string literals: const char(&)[N] / char(&)[N]
        template <class T>
        struct is_string_literal : std::false_type {};

        template <std::size_t N>
        struct is_string_literal<const char(&)[N]> : std::true_type {};

        template <std::size_t N>
        struct is_string_literal<char(&)[N]> : std::true_type {};

        template <class T>
        inline constexpr bool is_string_literal_v = is_string_literal<T>::value;

        // true for unsigned integer types (unsigned, uint32_t, etc.)
        template <class T>
        inline constexpr bool is_unsigned_int_v =
            std::is_integral_v<remove_cvref_t<T>> && std::is_unsigned_v<remove_cvref_t<T>>;

        template<class>
        inline constexpr bool dependent_false_v = false;
    }


    namespace Core {
        class Object;
    }

    enum class FieldType {
        Bool = 0,
        UInt = 1,
        Int  = 2,
        Real = 3,
        Max
    };

    enum class FieldArrayType {
        None     = 0, // Int
        Static   = 1, // Int[3], NOT SUPPORTED
        Dynamic  = 2, // Int[size], NOT SUPPORTED
        Flexible = 3, // Int[]
        Max
    };

    struct Field {
        FieldType field_type;
        size_t size;
        size_t offset;
        FieldArrayType field_array_type;
    };

    class Marshalling {
        public:
            Marshalling();
            static std::optional<Field> field_from_object(const std::shared_ptr<Core::Object>& object);
            bool add_fields_from_object(const std::shared_ptr<Core::Object>& object);
            void add_field(std::string name, FieldType field_type, size_t size, size_t offset, FieldArrayType field_array_type = FieldArrayType::None);
            void update_field(const std::string& name, size_t size, size_t offset);
            std::unique_ptr<Marshalling>& get_or_add_nested_marshalling(const std::string& name);
            void infer_buffer_size();
            void calculate_nested_buffer_sizes(size_t stride = 0);

            void clone_into(Marshalling& other);

            void set_buffer(uint8_t* buffer);
            uint8_t* get_buffer() const;

            void set_buffer_size(size_t buffer_size);
            size_t get_buffer_size() const;

            void set_size(size_t size);
            size_t get_size() const;

            void set_offset(size_t offset);
            size_t get_offset() const;

            void set_stride(size_t stride);
            size_t get_stride() const;

            void set_array_type(FieldArrayType array_type);
            FieldArrayType get_array_type() const;

            const std::unordered_map<std::string, Field>& get_field_map() const;
            const std::unordered_map<std::string, std::unique_ptr<Marshalling>>& get_nested_map() const;

            template <typename T, const char*...Args>
            std::optional<T> read();

            template <typename T>
            std::optional<T> read_real(const std::string& field_name);

            template <typename T>
            std::optional<T> read_real_indexed(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            T read_real_unsafe(const std::string& field_name);

            template <typename T>
            T read_real_indexed_unsafe(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            std::optional<T> read_int(const std::string& field_name);

            template <typename T>
            std::optional<T> read_int_indexed(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            T read_int_unsafe(const std::string& field_name);

            template <typename T>
            T read_int_indexed_unsafe(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            std::optional<T> read_uint(const std::string& field_name);

            template <typename T>
            std::optional<T> read_uint_indexed(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            T read_uint_unsafe(const std::string& field_name);

            template <typename T>
            T read_uint_indexed_unsafe(const std::string& field_name, size_t index, size_t field_index);

            std::optional<bool> read_bool(const std::string& field_name);
            std::optional<bool> read_bool_indexed(const std::string& field_name, size_t index, size_t field_index);
            bool read_bool_unsafe(const std::string& field_name);
            bool read_bool_indexed_unsafe(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            bool write_real(const std::string& field_name, T value);

            template <typename T>
            bool write_real_indexed(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            void write_real_unsafe(const std::string& field_name, T value);

            template <typename T>
            void write_real_indexed_unsafe(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            bool write_int(const std::string& field_name, T value);

            template <typename T>
            bool write_int_indexed(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            void write_int_unsafe(const std::string& field_name, T value);

            template <typename T>
            void write_int_indexed_unsafe(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            bool write_uint(const std::string& field_name, T value);

            template <typename T>
            bool write_uint_indexed(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            void write_uint_unsafe(const std::string& field_name, T value);

            template <typename T>
            void write_uint_indexed_unsafe(const std::string& field_name, T value, size_t index, size_t field_index);

            bool write_bool(const std::string& field_name, bool value);
            bool write_bool_indexed(const std::string& field_name, bool value, size_t index, size_t field_index);
            void write_bool_unsafe(const std::string& field_name, bool value);
            void write_bool_indexed_unsafe(const std::string& field_name, bool value, size_t index, size_t field_index);

            template <typename T, class Arg>
            std::optional<T> read(Arg&& arg) {
                if constexpr (MarshallingTemplateUtils::is_string_literal_v<Arg>) {
                    if constexpr (std::is_same_v<T, double>) {
                        return read_real<T>(arg);
                    }
                } else if constexpr (MarshallingTemplateUtils::is_unsigned_int_v<Arg>){
                    return std::nullopt;
                } else {
                    static_assert(MarshallingTemplateUtils::dependent_false_v<Arg>, "Argument must be string literal or unsigned integer");
                }
                return std::nullopt;
            }

            template <typename T, class Arg, class... Rest>
            std::optional<T> read(Arg&& first, Rest&&... rest) {
                if constexpr (MarshallingTemplateUtils::is_string_literal_v<Arg>) {
                    auto it = m_nested_map.find(std::string(first));
                    if (it == m_nested_map.end()) return std::nullopt;
                    auto& nested_marshalling = get_or_add_nested_marshalling(first);
                    return nested_marshalling->template read<T, Rest...>(std::forward<Rest>(rest)...);
                } else if constexpr (MarshallingTemplateUtils::is_unsigned_int_v<Arg>) {
                    return std::nullopt;
                } else {
                    static_assert(MarshallingTemplateUtils::dependent_false_v<Arg>, "Argument must be string literal or unsigned integer");
                }
                return std::nullopt;
            }

        private:
            size_t m_size;
            size_t m_offset;
            size_t m_stride;
            FieldArrayType m_array_type;
            uint8_t* m_buffer;
            size_t m_buffer_size;
            std::unordered_map<std::string, Field> m_field_map;
            std::unordered_map<std::string, std::unique_ptr<Marshalling>> m_nested_map;
    };
}
