#pragma once

#include <cstdint>
#include <limits>
#include <memory>
#include <string>
#include <vector>
#include <openplx/control_export.h>

namespace openplx {

    namespace Physics::Signals {
        class SignalInterface;
        class Output;
        class Input;
    }

    class Marshalling;
    class DetachedControl;
    class ControlDispatch;
    class ControlInterfaceImpl;

    /**
     * A ControlInterface is a wrapper for reading and writing to a set of input and output controls.
     *
     * It can operate on two modes: attached or detached
     *
     * In attached mode a ControlDispatch instance is injected to the constructor which allows the ControlInterface
     * to read and write directly to a simulation.
     *
     * In detached mode `nullptr` is passed to the constructor and the ControlInterface can only be used to decode or
     * encode control buffers that come from or will be sent to an attached ControlInterface.
     *
     * Reading or writing to a ControlInterface is done exclusively through raw memory buffers with a uint8_t* and a size_t
     * for the buffer size. This delegates control of memory allocation to the end user. For simpler reading and writing
     * there are higher level interface classes like HeapControlInterface and JsonControlInterface that wraps a ControlInterface
     * that have a simpler interface.
     *
     * For detached use cases where the original .openplx file is not accessible (usually because some plugin is unavailable)
     * it is possible to load the controls from JSON. The application that has access to the full model (and all necessary
     * plugins) can serialize a ControlInterface to JSON which can then be sent to the application without access and a
     * ControlInterface can still be used.
     */
    class OPENPLX_CONTROL_EXPORT ControlInterface {
        public:

            static constexpr uint32_t INVALID_KEY = std::numeric_limits<uint32_t>::max();
            static constexpr uint32_t SEPARATOR_KEY = std::numeric_limits<uint32_t>::max() - 1;

            /**
             * Dispatch can be nullptr if we are in a detached state.
             */
            ControlInterface(std::shared_ptr<ControlDispatch> dispatch);
            ~ControlInterface();

            /**
             * Adds the input and output controls from a SignalInterface to the control interface. Unique keys are allocating by
             * starting from 1 and incrementing everytime a control is added.
             * @param signal_interface The signal interface to add controls from.
             */
            void add_controls_from_signal_interface(const std::shared_ptr<Physics::Signals::SignalInterface>& signal_interface);

            /**
             * Adds an output control to the control interface.
             * @param key A unique (in the scope of this control interface) 32 bit unsigned integer identifying the control.
             * @param name A string representing the control (does not need to be unique).
             * @param output A reference to the OpenPLX Output control instance.
             */
            void add_output(uint32_t key, std::string name, std::shared_ptr<Physics::Signals::Output> output);

            /**
             * Adds an input control to the control interface.
             * @param key A unique (in the scope of this control interface) 32 bit unsigned integer identifying the control.
             * @param name A string representing the control.
             * @param output A reference to the OpenPLX Input control instance.
             */
            void add_input(uint32_t key, std::string name, std::shared_ptr<Physics::Signals::Input> input);

            /**
             * @return A json representation of the control interface.
             */
            std::string to_json() const;

            /**
             * Adds controls from JSON data.
             * @param json_str A string with the JSON encoded control interface.
             */
            void add_controls_from_json(const std::string& json_str);

            /**
             * @param name The name of the input or output control.
             * @return The key associated with that control.
             */
            uint32_t lookup_control_key_from_name(const std::string& name);

            /**
             * @param uuid The uuid of the input or output control.
             * @return The key associated with that control.
             */
            uint32_t lookup_control_key_from_uuid(const std::string& uuid);

            /**
             * @return a list of all input control keys
             */
            std::vector<uint32_t> lookup_input_control_keys();

            /**
             * @return a list of all input control keys
             */
            std::vector<uint32_t> lookup_output_control_keys();

            /**
             * Resolves and sorts a set of input control keys, only needed when using
             * virtual controls.
             * @return A new list where virtual control dependencies are resolved.
             */
            std::vector<uint32_t> resolve_input_keys(const std::vector<uint32_t>& keys);

            /**
             * Resolves and sorts a set of output control keys, only needed when using
             * virtual controls.
             * @return A new list where virtual control dependencies are resolved.
             */
            std::vector<uint32_t> resolve_output_keys(const std::vector<uint32_t>& keys);

            /**
             * @param key The key to lookup.
             * @return A DetachedControl instance representing the control.
             */
            std::shared_ptr<DetachedControl> lookup_detached_control(uint32_t key);

            /**
             * Calculates how many bytes is necessary to create a control buffer for writing to a subset of the
             * available controls.
             * @param resolved_keys The subset of controls to use given by their keys.
             * @return The buffer size necessary to create a control buffer to write to the controls.
             */
            size_t calculate_input_buffer_size(const std::vector<uint32_t>& resolved_keys);

            /**
             * Writes the necessary headers to a input buffer, this is necessary before calling prepare_marshalling.
             * @param requested_keys The subset of controls to use given by their keys in order they should appear in the payload.
             * @param resolved_keys The result after resolving the requested key, usually by calling resolve_output_keys, sorted in dependency order
             * @param buffer The control buffer where data will be written.
             * @param buffer_size The size of the control buffer.
             */
            void prepare_input_buffer(
                const std::vector<uint32_t>& requested_keys,
                const std::vector<uint32_t>& resolved_keys,
                uint8_t* buffer, size_t buffer_size);

            /**
             * Prepares a control for reading or writing and returns the corresponding marshalling instance.
             * @param key The key of the control.
             * @param buffer The control buffer where data will be read or written.
             * @param buffer_size The size of the control buffer.
             * @return The marshalling instance that can be used for reading or writing.
             */
            std::shared_ptr<Marshalling> prepare_marshalling(uint32_t key, uint8_t* buffer, size_t buffer_size);

            /**
             * Calculates how many bytes is necessary to create a control buffer for reading from a subset of the
             * available controls.
             * @param resolved_keys The subset of controls to use given by their keys.
             * @return The buffer size necessary to create a control buffer to read from the controls.
             */
            size_t calculate_output_buffer_size(const std::vector<uint32_t>& resolved_keys);

            /**
             * Writes the necessary headers to a output control buffer, this is necessary before calling prepare_marshalling.
             * @param requested_keys The subset of controls to use given by their keys in order they should appear in the payload.
             * @param resolved_keys The result after resolving the requested key, usually by calling resolve_output_keys, sorted in dependency order
             * @param buffer The control buffer where data will be read.
             * @param buffer_size The size of the control buffer.
             */
            void prepare_output_buffer(
                const std::vector<uint32_t>& requested_keys,
                const std::vector<uint32_t>& resolved_keys,
                uint8_t* buffer, size_t buffer_size);

            /**
             * Prepares the controls, must be called after all inputs and outputs
             * have been registered.
             */
            void prepare_controls();

            /**
             * Propagates data from virtual inputs to their targets in the same buffer.
             * @param buffer The control buffer where data will be written.
             * @param buffer_size The size of the control buffer.
             * @return true if the operation was successful, false otherwise.
             */
            bool resolve_virtual_inputs(uint8_t* buffer, size_t buffer_size);

            /**
             * Propagates data to virtual outputs from their sources in the same buffer.
             * @param buffer The control buffer where data will be written.
             * @param buffer_size The size of the control buffer.
             * @return true if the operation was successful, false otherwise.
             */
            bool resolve_virtual_outputs(uint8_t* buffer, size_t buffer_size);

            /**
             * Dispatches a input buffer to the simulation which will update the state of the simulation.
             * Note: This only works if the control interface is in attached mode.
             * @param buffer The control buffer where data has been written.
             * @param buffer_size The size of the control buffer.
             * @return true if the dispatch was successful, false otherwise.
             */
            bool dispatch_input_buffer(uint8_t* buffer, size_t buffer_size);

            /**
             * Dispatches an output buffer to the simulation which will read the state of the simulation and write it into the buffer.
             * Note: This only works if the control interface is in attached mode.
             * @param buffer The control buffer.
             * @param buffer_size The size of the control buffer.
             * @return true if the dispatch was successful, false otherwise.
             */
            bool dispatch_output_buffer(uint8_t* buffer, size_t set_buffer_size);

            /**
             * Bypasses the error and bounds checks and tries to read a single double field from an output
             * control fast. Undefined behavior if the key does not match an output which outputs a single real
             * field.
             * @param key The key of the output control
             * @return The value read from that output control as a double
             */
            double read_double_unsafe(uint32_t key);

        private:
            ControlInterfaceImpl* m_impl;
    };
}
