/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agx/config/AGX_USE_AGXTERRAIN.h>
#include <agxTerrain/export.h>

#include <array>
#include <agx/Vec3.h>
#include <agx/agx_vector_types.h>
#include <agxTerrain/TerrainContact.h>

#include <agxTerrain/FailureSurface.h>
#include <agxTerrain/FailureSurfaceAggregate.h>
#include <agxCollide/Contacts.h>
#include <agxRender/Color.h>



namespace agxTerrain
{
  namespace GroundCollapse
  {
    /*
    The position and force of an external load.
    */
    struct AGXTERRAIN_EXPORT LoadPoint
    {
      agx::Vec3 localPoint;
      agx::Vec3 verticalForce;
    };
    using LoadPointVector = agx::VectorPOD<LoadPoint>;

    /*
    Data needed to create a collapse simulation.
    */
    struct AGXTERRAIN_EXPORT ContactData
    {
      FailureSurfaceAggregateRef m_aggregate;
      FailureSurfaceRef m_surface;
      agxCollide::LocalContactPointVector m_contactPoints;
      agx::ContactMaterial* m_contactMaterial;
      GroundCollapse::LoadPointVector m_loadPoints;

      agx::Real getContactArea() const
      {
        return m_aggregate->getVoxelContactArea();
      }

      agx::Real getTotalLoadForce() const
      {
        agx::Real totalLoadForce = 0;
        for (auto& p : m_loadPoints)
          totalLoadForce += p.verticalForce.z();

        return totalLoadForce;
      }

      bool isValid() const
      {
        return m_aggregate != nullptr && m_surface != nullptr;
      }
    };

    using ContactDataVector = agx::Vector<ContactData>;


    AGXTERRAIN_EXPORT void collapseSurfaceToGranular( agxTerrain::Terrain* terrain,
                                                      const FailureSurface* failureSurface );

    AGXTERRAIN_EXPORT agx::AffineMatrix4x4 constructSurfaceLocalTransform( const agxTerrain::Terrain* terrain,
                                                                           const GroundCollapse::FailureSurface* failureSurface );

    DOXYGEN_START_INTERNAL_BLOCK()
    namespace SlopeUtils
    {
      struct AGXTERRAIN_EXPORT SlopeData
      {
        agx::Vec3 beforeTheSlope;
        agx::Vec3 startOfSlope;
        agx::Vec3 endOfSlope;
        agx::Vec3 slopeDirection;
        bool isValid = false;
      };

      using SearchLine = agx::Vec3Vector;
      using SearchLineVector = agx::Vector<SearchLine>;

      struct AGXTERRAIN_EXPORT EndPoints
      {
        agx::Vec3Vector pointsA;
        agx::Vec3Vector pointsB;

        bool isValid()
        {
          return pointsA.size() > 0 && pointsB.size();
        }
      };
      /*
      Search for a slope around a given position on the terrain.

      This is done by shooting rays in a circle around the given position and projecting them
      onto the terrain surface. Along each ray, the projected points are then examined to see
      whether they have a big enough gradient or not.
      \param terrain - The terrain where you want to find a slope.
      \param position - The position to start the search from.
      \param numberOfRays - How many rays are spread out around the starting position. Default: 40.
      \param numberOfTraversalPoints - How many points each ray will be discretized into. Default: 200.
      \param searchLineLength - Max length of the rays. Default: 50.0.
      \param debugRender - Enable/disable the debug rendering of the slope search rays. Default: false.
      \return SlopeData struct describing the steepest slope in the vicinity of the given position.
      */
      AGXTERRAIN_EXPORT SlopeData searchForSlope( agxTerrain::Terrain* terrain,
                                                  agx::Vec3 position,
                                                  agx::UInt numberOfRays = 40,
                                                  agx::UInt numberOfTraversalPoints = 200,
                                                  agx::Real searchLineLength = 50.0,
                                                  bool debugRender = false );

      /*
      Create the search line rays around a given position.
      \param position - The position to start the search from.
      \param numberOfRays - How many search lines should be evenly distributed
      in a 360 degree arc around the starting position.
      \param numberOfTraversalPoints - How many points each search line will be
      discretized into
      \param searchLineLength - Max length of the search lines.
      \return a vector containing the search lines.
      */
      AGXTERRAIN_EXPORT SearchLineVector createSearchLines( agxTerrain::Terrain* terrain,
                                                            agx::Vec3 position,
                                                            agx::UInt numberOfRays,
                                                            agx::UInt numberOfTraversalPoints,
                                                            agx::Real searchLineLength );
      /*
      Find slope data along the search line. Only filters for the first slope from the
      starting position in the search line.
      \param terrain - the terrain to find a slope on.
      \param line - a vector of positions along the terrain surface.
      \return struct defining a slope along the search line.
      */
      AGXTERRAIN_EXPORT SlopeData createSlopeData( Terrain* terrain, SearchLine line );

      /*
      Filter the search lines for those that are steep enough to fulfill the given gradient limit.
      \param lines - all search lines to filter through.
      \param gradLimit - the gradient limit on the search line.
      \return the search lines that are steep enough.
      */
      AGXTERRAIN_EXPORT SearchLineVector filterSteepLines( agxTerrain::Terrain* terrain, SearchLineVector lines, agx::Real gradLimit = 0.1);

      /*
      Compare the given search lines and find the steepest one of them.
      \param lines - all search lines to filter through.
      \return the steepest search line.
      */
      AGXTERRAIN_EXPORT SearchLine findSteepestLine(agxTerrain::Terrain* terrain, SearchLineVector lines );

      /*
      Generate end points that intersect above and on/below the slope in the
      terrain for the different failure surfaces.
      \param terrain - the terrain to generate end points on.
      \param slope - data that describes the slope on the given terrain.
      \param contacts - terrain contacts that the end points are partitioned with respect to.
      \param numPointsA - how many points are on/below the slope.
      \param numPointsB how many points are above the slope.
      \param upperPointSpanOffset- a value [0, 1] to offset the starting end point away from the slope. Default: 0.1.
      \param lowerPointSpanOffset - a fraction [0, 1] (between slope end points) which determines from where the lower points are generated. Default: 0.25.
      \param lowerPointSpanScaleFactor - the scale factor of the distance span of the lower points on the slope. Default: 1.25.

      \return end points that can be used to generate failure surfaces on the slope.
      */
      AGXTERRAIN_EXPORT EndPoints createSamplePoints( agxTerrain::Terrain* terrain,
                                                      const SlopeData& slope,
                                                      const agxTerrain::TerrainContactPtrVector& contacts,
                                                      agx::UInt numPointsA,
                                                      agx::UInt numPointsB,
                                                      agx::Real upperPointSpanOffset = 0.1,
                                                      agx::Real lowerPointSpanOffset = 0.25,
                                                      agx::Real lowerPointSpanScaleFactor = 1.25 );

      /*
      Construct failure surfaces on the terrain from the end points and number
      of radii. If n_a and n_b are points on/below and above the slope respectively,
      and n_r is how many different radii to use, the number of constructed failure
      surfaces will be equal to n_a * n_b * n_r.
      \param terrain - the terrain to construct failure surfaces on.
      \param endPoints - the points on/below and above the slope (x_a and x_b).
      \param radiusNumber - how many different radii to use (n_r).
      \return A vector of failure surfaces.
      */
      AGXTERRAIN_EXPORT FailureSurfacePtrVector constructFailureSurfaces( agxTerrain::Terrain* terrain,
                                                                          const EndPoints& endPoints,
                                                                          agx::UInt radiusNumber );
    }

    using PointConfiguration = agx::VectorPOD<int>;

    /*
    Generate contact points between the failure surface and the rest of the terrain.
    \param failrueSurface - the failure surface.
    \param terrain - the terrain.
    \param pointConfiguration - configuration of the contact points position on the
    interface between the failure surface and the terrain.
    \param shrinkFactor - factor that shrinks the segments.
    \param enableDebugRendering - enable/disable debug rendering of the contact point
    generation. Default: false.
    \return vector containing the generated contact points.
    */
    AGXTERRAIN_EXPORT agxCollide::LocalContactPointVector generateFailureSurfaceContactPoints(
      const FailureSurface* failureSurface,
      agxTerrain::Terrain* terrain,
      const PointConfiguration& pointConfiguration,
      const agx::Vec2& shrinkFactor,
      agx::UInt numIntersectionPoints,
      bool enableDebugRendering=false );

    /*
    Find intersecting solid voxels in a radius around a world position on a terrain.
    \param position - world position to find intersecting voxels around.
    \param radius - the radius to find intersecting voxels within.
    \param terrain - the terrain to find intersecting voxels on.
    \return voxel indices of the intersecting solid voxels inside the specified terrain.
    */
    AGXTERRAIN_EXPORT agx::Vec3iVector findIntersectingVoxels(agx::Vec3 position,
                                                              agx::Real radius,
                                                              const agxTerrain::Terrain* terrain);

    DOXYGEN_END_INTERNAL_BLOCK()
  }
}