/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agxTerrain/TerrainContact.h>
#include <agxTerrain/GroundCollapseUtils.h>
#include <agxTerrain/CollapseSimulation.h>

namespace agxTerrain
{
  class Terrain;
  namespace GroundCollapse
  {
    // Forward declarations
    class CollapseSettings;
    class GroundCollapseController;
    class GroundCollapseResult;
    using GroundCollapseResultRef = agx::ref_ptr< GroundCollapseResult >;

    DOXYGEN_START_INTERNAL_BLOCK()
    AGX_DECLARE_POINTER_TYPES(GroundCollapseCalculator);

    /*
    The GroundCollapseCalculator is responsible for setting up and running everything related to the ground collapse algorithm.

    From the user's perspective, the algorithm is enabled via TerrainProperties and settings are configured in the GroundCollapseController.
    */
    class AGXTERRAIN_EXPORT GroundCollapseCalculator : public agx::Referenced
    {
    public:

      /*
      Constructor given a controller.
      */
      GroundCollapseCalculator(GroundCollapseController* controller);

      /*
      Calculates the ground collapse inside the given zones. Initiated from the controller.
      */
      GroundCollapse::GroundCollapseResultRef calculateGroundCollapse();

      /*
      Calculates the ground collapse inside the given zones but in parallel.
      */
      GroundCollapse::GroundCollapseResultRef calculateGroundCollapseParallel();

      /*
      Construct soil aggregates from the given failure surfaces and terrain.
      \param terrain - The terrain where the soil aggregates will be created from.
      \param contactData - Vector which stores necessary data for the ground collapse simulation.
      \param failureSurfaces - The failure surfaces inside the given terrain to construct soil aggregates from.
      */
      void constructSoilAggregates( agxTerrain::Terrain* terrain,
                                    ContactDataVector& contactData,
                                    const FailureSurfacePtrVector& failureSurfaces );
      /*
      Prepare the contact data for a failure surface simulation.
      \param terrain - the terrain where the failure surface simulation takes place.
      \param contacts - the terrain contacts which exert an external force upon the soil, and may make it collapse.
      \param contactData - Vector which stores necessary data for the ground collapse simulation.
      */
      void prepareContactData( agxTerrain::Terrain* terrain,
                               const TerrainContactPtrVector& contacts,
                               ContactDataVector& contactData );

      /*
      Adds the necessary contact data for a failure surface to the ground collapse simulation.
      */
      int addContactDataToSim( ContactDataVector& contactData );

      /*
      * Solve wedges in parallel.
      */
      void solveWedgeParallel( FailureSurfaceResultVector& results );

      /*
      Set the contact material to use in the simulation, either from the materials in the contact points or the custom material.
      */
      agx::ContactMaterial* allocateContactMaterial(
        agxTerrain::Terrain* terrain,
        agx::Real contactArea,
        const agxCollide::LocalContactPointVector& materialSamplingPoints );

      /*
      Calculate whether the given failure surface inside the given terrain fails under the load from the given load points.
      \param terrain - the terrain the failure surface is inside.
      \param surface - the failure surface to solve for.
      \param loadPoints - the load exerted on the surface.
      \return the result from a ground collapse simulation solve.
      */
      FailureSurfaceResult solveWedge( agxTerrain::Terrain* terrain,
                                       FailureSurface* surface,
                                       const LoadPointVector& loadPoints );

      /*
      Find the mid point weighted by force magnitude from the supplied terrain contacts.
      \param contacts - The terrain contacts to find the weighted mid point from.
      \return the force weighted mid point of all the point forces in the contacts.
      */
      agx::Vec3 findWeightedMidPoint( const TerrainContactPtrVector& contacts);

      /*
      Translate the terrain contacts into load points - disregard those that are outside the surface.
      \param upDirection - the terrain normal
      \param surface - the failure surface
      \param contacts - terrain contacts in the zone(s)
      \return load points on the surface
      */
      LoadPointVector getLoadPointsOnSurface( agx::Vec3 upDirection, const FailureSurface* surface, const TerrainContactPtrVector& contacts ) const;

    protected:
      void writeResultData( FailureSurfaceResult& result,
                            const ContactData& cd,
                            CollapseSettings* settings );

    private:
      GroundCollapseController*     m_controller;
      CollapseSimulationRef         m_collapseSimulation;
      agx::ParallelTaskRef          m_intersectionTask;
    };

    DOXYGEN_END_INTERNAL_BLOCK()
  }
}