/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/SystemNode.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( ISystemNodeProxy );
  AGX_DECLARE_VECTOR_TYPES( ISystemNodeProxy );
  AGX_DECLARE_POINTER_TYPES( SystemNodeProxy );
  AGX_DECLARE_VECTOR_TYPES( SystemNodeProxy );

  /**
  Interface for system node behavioral implementations intended to have multiple states/instances
  (SystemNodeProxy/ies) in the system graph. These implementations only store behavioral
  implementation parameters, while leaving the runtime state to each instance/proxy in the system
  graph.
  */
  class AGXSENSOR_EXPORT ISystemNodeProxy : public agx::Referenced, public agxStream::Serializable
  {
    public:
      using CallbackData = SystemNode::CallbackData;

    public:
      /**
      \return new, unique, system node (proxy) of this implementation
      */
      virtual SystemNodeProxyRef createProxy() = 0;

      /**
      Called when this system node has been added to a sensor environment.
      \param proxy - system graph instance (proxy) carrying this implementation
      \return true if successful and is ready to receive runtime calls
      */
      virtual bool addNotification( SystemNodeProxy& proxy );

      /**
      Called when this system node is being removed from a sensor environment.
      \param proxy - system graph instance (proxy) carrying this implementation
      */
      virtual void removeNotification( SystemNodeProxy& proxy );

      /**
      Called when \p child has been added to this parent.
      \param proxy - system graph instance (proxy) carrying this implementation
      \param child - child that has been added
      */
      virtual void onChildAdded( SystemNodeProxy& proxy, SystemNode& child );

      /**
      Called when \p child has been removed from this parent.
      \param proxy - system graph instance (proxy) carrying this implementation
      \param child - child that has been removed
      */
      virtual void onChildRemoved( SystemNodeProxy& proxy, SystemNode& child );

      /**
      Called when an object has been added to the sensor environment.
      \param proxy - system graph instance (proxy) carrying this implementation
      \param instance - object that was added to the sensor environment
      \return true if the object is consumed by the system, false if this system node doesn't
              handle/support this object.
      */
      virtual bool onAdd( SystemNodeProxy& proxy, agx::Referenced& instance );

      /**
      Called when an object has been removed from the sensor environment.
      \param proxy - system graph instance (proxy) carrying this implementation
      \param instance - object that was removed from the sensor environment
      \return true if the object is consumed by the system, false if this system node doesn't
              handle/support this object.
      */
      virtual bool onRemove( SystemNodeProxy& proxy, agx::Referenced& instance );

      /**
      Synchronization from the main thread. Read data from objects/simulation should be done here
      before execute/stepping is made separate from the main thread.
      \param proxy - system graph instance (proxy) carrying this implementation
      \param data - runtime step data
      */
      virtual void synchronize( SystemNodeProxy& proxy, const CallbackData& data );

      /**
      Execute this system, integrating it \p data.dt forward in time. This call is made from a
      thread separate from the main/simulation thread so it's not valid to read data from the
      objects being simulated.
      \param proxy - system graph instance (proxy) carrying this implementation
      \param data - runtime step data
      */
      virtual void execute( SystemNodeProxy& proxy, const CallbackData& data );

      /**
      Complete the execution of this system, finishing the integration forward in time. This call is
      made from a thread separate from the main/simulation thread so it's not valid to read data
      from the objects being simulated.
      \param proxy - system graph instance (proxy) carrying this implementation
      */
      virtual void complete( SystemNodeProxy& proxy );

      /**
      If necessary assemble, and fetch results related to this system node after a step by
      \p data.dt has been completed. This call is made from the main thread, post-step, and thus
      allows the latest possible data from objects/simulation to be read and processed.
      \param proxy - system graph instance (proxy) carrying this implementation
      \param data - runtime step data
      */
      virtual void result( SystemNodeProxy& proxy, const CallbackData& data );

      /**
      The sensor environment is being put down, clean everything.
      \param proxy - system graph instance (proxy) carrying this implementation
      */
      virtual void cleanup( SystemNodeProxy& proxy ) = 0;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSensor::ISystemNodeProxy );

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  /**
  Proxy system node, representing an instance of an ISystemNodeProxy in the system graph. This
  allows one behavioral implementation (an ISystemNodeProxy) to have multiple states/instances
  in the system graph, while common parameters are still shared through the ISystemNodeProxy.
  */
  class AGXSENSOR_EXPORT SystemNodeProxy : public SystemNode
  {
    public:
      /**
      Construct a system node proxy instance with the given \p implementation.
      \param implementation - behavioral implementation for this proxy (should not be nullptr)
      */
      SystemNodeProxy( ISystemNodeProxy* implementation );

      /**
      \return true if this node implements the given type, otherwise false
      */
      template<typename T>
      bool implements() const;

      /**
      \return the behavioral implementation behind this system node proxy
      */
      ISystemNodeProxy* getImplementation();

      /**
      \return the behavioral implementation behind this system node proxy
      */
      const ISystemNodeProxy* getImplementation() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      SystemNodeProxy();

      virtual bool addNotification() override;
      virtual void removeNotification() override;
      virtual void onChildAdded( SystemNode& child ) override;
      virtual void onChildRemoved( SystemNode& child ) override;
      virtual bool onAdd( agx::Referenced& instance ) override;
      virtual bool onRemove( agx::Referenced& instance ) override;
      virtual void synchronize( const CallbackData& data ) override;
      virtual void execute( const CallbackData& data ) override;
      virtual void complete() override;
      virtual void result( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::SystemNodeProxy );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      ISystemNodeProxyRef m_implementation;
  };

  template<typename T>
  bool SystemNodeProxy::implements() const
  {
    return m_implementation->is<T>();
  }
}
