/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/export.h>
#include <agxSensor/MonoaxialSignalSystemNode.h>

#include <agxStream/Serializable.h>

#include <agx/Range.h>
#include <agx/Referenced.h>

#ifdef _WIN32
#undef ABSOLUTE
#endif

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( EncoderModel );

  /**
  Base encoder model describing the fundamental parameters, such as cycle range and accumulation
  mode, of rotary encoders, linear encoders and faux potentiometers.
  \note The configuration in this model is shared and synchronized across all instances.
  */
  class AGXSENSOR_EXPORT EncoderModel : public agx::Referenced, public agxStream::Serializable
  {
    public:
      enum Mode : int8_t
      {
        /** Wrap turns and do not increment total output value. */
        INCREMENTAL,
        /** Single turn, absolute range. */
        ABSOLUTE
      };

    public:
      /**
      Construct an encoder model with the specified parameters.
      \param mode - accumulation mode of the encoder
      \param range - range of one encoder turn
      \param outputModifiers - list of modifiers to apply to the output of the encoder
      */
      EncoderModel( Mode mode = INCREMENTAL,
                    agx::RangeReal range = agx::RangeReal( 0.0, 2.0 * agx::PI ),
                    IMonoaxialSignalSystemNodeRefVector outputModifiers =
                      IMonoaxialSignalSystemNodeRefVector() );

      /**
      Set the encoder accumulation mode.
      \param mode - accumulation mode of the encoder
      */
      void setMode( Mode mode );

      /**
      \return accumulation mode of the encoder
      */
      Mode getMode() const;

      /**
      Set the range of one encoder turn.
      \param range - range of one encoder turn
      */
      void setRange( agx::RangeReal range );

      /**
      \return range of one encoder turn
      */
      agx::RangeReal getRange() const;

      /**
      \return the list of output modifiers to apply to the output
      */
      const IMonoaxialSignalSystemNodeRefVector& getOutputModifiers() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~EncoderModel() = default;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::EncoderModel );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      Mode m_mode;
      agx::RangeReal m_range;
      IMonoaxialSignalSystemNodeRefVector m_outputModifiers;
  };
}
