/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <list>
#include <memory>
#include <unordered_set>
#include <agx/BallJoint.h>
#include <agx/CylindricalJoint.h>
#include <agx/Hinge.h>
#include <agx/LockJoint.h>
#include <agx/Material.h>
#include <agx/MergedBody.h>
#include <agx/Prismatic.h>
#include <agx/RigidBody.h>
#include <agx/SingleControllerConstraint1DOF.h>
#include <agx/SlackCylindricalJoint.h>
#include <agx/SlackHingeJoint.h>
#include <agx/SlackLockJoint.h>
#include <agx/SlackPrismaticJoint.h>
#include <agx/DistanceJoint.h>
#include <agx/Vector.h>
#include <agxCollide/Capsule.h>
#include <agxCollide/Convex.h>
#include <agxDriveTrain/CombustionEngine.h>
#include <agxDriveTrain/Differential.h>
#include <agxDriveTrain/DryClutch.h>
#include <agxDriveTrain/Engine.h>
#include <agxDriveTrain/Gear.h>
#include <agxDriveTrain/GearBox.h>
#include <agxDriveTrain/Shaft.h>
#include <agxDriveTrain/TorqueConverter.h>
#include <agxModel/SuctionGripper.h>
#include <agxPowerLine/Actuator1DOF.h>
#include <agxPowerLine/Connector.h>
#include <agxPowerLine/PowerLine.h>
#include <agxPowerLine/RotationalUnit.h>
#include <agxPowerLine/Unit.h>
#include <agxSDK/Assembly.h>
#include <agxSDK/EventListener.h>
#include <agxTerrain/Shovel.h>
#include <agxTerrain/Terrain.h>
#include <agxVehicle/Steering.h>
#include <agxVehicle/Track.h>
#include <agxVehicle/WheelJoint.h>
#include <openplx/DriveTrain/AutomaticClutch.h>
#include <openplx/DriveTrain/EmpiricalEngine.h>
#include <openplx/DriveTrain/EmpiricalTorqueConverter.h>
#include <openplx/DriveTrain/Gear.h>
#include <openplx/DriveTrain/GearBox.h>
#include <openplx/DriveTrain/HingeActuator.h>
#include <openplx/DriveTrain/ManualClutch.h>
#include <openplx/DriveTrain/MeanValueEngine.h>
#include <openplx/DriveTrain/PrismaticActuator.h>
#include <openplx/DriveTrain/TorqueMotor.h>
#include <openplx/Error.h>
#include <openplx/ErrorReporter.h>
#include <openplx/Math/Vec3.h>
#include <openplx/Object.h>
#include <openplx/Physics/Geometries/Material.h>
#include <openplx/Physics/Interactions/Dissipation/DefaultDissipation.h>
#include <openplx/Physics/Interactions/Flexibility/DefaultFlexibility.h>
#include <openplx/Physics/Interactions/Interaction.h>
#include <openplx/Physics/Interactions/SurfaceContact/Model.h>
#include <openplx/Physics/KinematicLock.h>
#include <openplx/Physics1D/Bodies/RotationalBody.h>
#include <openplx/Physics3D/Bodies/RigidBody.h>
#include <openplx/Physics3D/Geometries/Box.h>
#include <openplx/Physics3D/Geometries/Capsule.h>
#include <openplx/Physics3D/Geometries/ConvexMesh.h>
#include <openplx/Physics3D/Geometries/Cylinder.h>
#include <openplx/Physics3D/Geometries/Sphere.h>
#include <openplx/Physics3D/Interactions/Cylindrical.h>
#include <openplx/Physics3D/Interactions/Hinge.h>
#include <openplx/Physics3D/Interactions/LinearRange.h>
#include <openplx/Physics3D/Interactions/LinearSpring.h>
#include <openplx/Physics3D/Interactions/Prismatic.h>
#include <openplx/Physics3D/Interactions/RotationalRange.h>
#include <openplx/Physics3D/Interactions/SpringInteraction1DOF.h>
#include <openplx/Physics3D/Interactions/TorsionSpring.h>
#include <openplx/Physics3D/Interactions/VelocityMotor.h>
#include <openplx/Physics3D/Interactions/Distance.h>
#include <openplx/Robotics/EndEffectors/SuctionCup.h>
#include <openplx/Robotics/EndEffectors/VacuumGripper.h>
#include <openplx/Simulation/CollisionGroup.h>
#include <openplx/Simulation/DisableCollisionPair.h>
#include <openplx/Simulation/Simulation_all.h>
#include <openplx/Terrain/Terrain.h>
#include <agxModel/TwoBodyTire.h>
#include <agxModel/OneBodyTire.h>
#include <openplx/UuidGenerator.h>
#include <openplx/Vehicles/Tracks/System.h>
#include <agxOpenPLX/AgxCache.h>
#include <agxOpenPLX/AgxMetadata.h>
#include <agxOpenPLX/OpenPlxDriveTrainMapper.h>
#include <agxOpenPLX/OpenPlxMaterialManager.h>
#include <agxOpenPLX/OpenPlxSensorsMapper.h>
#include <agxOpenPLX/OpenPlxVehicleMapper.h>
#include <agxOpenPLX/export.h>

namespace agxopenplx
{

  struct ElementaryConstraintDOF
  {
      agx::ElementaryConstraintRef ec;
      uint32_t row;
  };

  /**
   * @brief Maps OpenPLX objects to AGX objects
   *
   */
  class AGXOPENPLX_EXPORT OpenPlxToAgxMapper
  {
    public:
      /**
       * @param agx_cache Cache to use for caching AGX objects.
       * @param agx_metadata Metadata storage for AGX objects.
       * @param material_manager Material manager to use for merging materials.
       * @param map_drivetrain Whether or not to map perform mapping for the drivetrain models.
       * @param map_vehicle Whether or not to map perform mapping for the vehicle models.
       * @param sensors_mapper Sensors mapper to use for mapping parts of the sensors bundle.
       * @param error_reporter Error reporter to use for reporting errors.
       */
      OpenPlxToAgxMapper(
        agxSDK::Simulation* simulation, std::shared_ptr<AgxCache> agx_cache = nullptr,
        std::shared_ptr<AgxMetadata> agx_metadata = nullptr,
        std::shared_ptr<OpenPlxMaterialManager> material_manager = nullptr, bool map_drivetrain = true,
        bool map_vehicle = true, std::shared_ptr<OpenPlxSensorsMapper> sensors_mapper = nullptr,
        std::shared_ptr<openplx::ErrorReporter> error_reporter = std::make_shared<openplx::ErrorReporter>());
      /**
       * @brief Map an OpenPLX object to an AGX Assembly
       * @param object The OpenPLX object to map, if plain Core::Object then no mapping will occur.
       */
      agxSDK::AssemblyRef mapObject(std::shared_ptr<openplx::Core::Object> object);
      std::shared_ptr<openplx::ErrorReporter> getErrorReporter();

      agxPowerLine::PowerLineRef getPowerLine() const;

      agxSDK::Simulation* getSimulation() const;
      std::shared_ptr<AgxCache> getAgxCache() const;
      std::shared_ptr<AgxMetadata> getAgxMetadata() const;

      std::shared_ptr<OpenPlxSensorsMapper> getOpenPlxSensorsMapper() const;

      agx::RigidBodyRef lookupAgxRigidBody(
        const std::shared_ptr<openplx::Physics3D::Bodies::RigidBody>& rigid_body) const;
      agxTerrain::TerrainRef lookupAgxTerrain(const std::shared_ptr<openplx::Terrain::Terrain>& b_terrain) const;
      const agx::Vector<agxSDK::EventListenerRef>& getEventListeners() const;
      const std::unordered_map<std::shared_ptr<openplx::Vehicles::Tracks::System>, agxVehicle::TrackRef>& getTracksMap()
        const;

    private:

      static constexpr const char* UuidRenderDataIdentifier = "render_data";
      static constexpr const char* UuidShapeIdentifier = "shape";
      static constexpr const char* UuidVisualShapeIdentifier = "visual_shape";
      static constexpr const char* UuidSingleControllerConstraintIdentifier = "agx::SingleControllerConstraint1DOF";

      template <typename T>
      static void regenerateUuid(const openplx::Core::Object& openplx_object, agx::ref_ptr<T>& agx_obj)
      {
        if (agx_obj != nullptr) {
          auto uuid = openplx::UuidGenerator::generateUuidv5(
            openplx_object.getUuid(), openplx_object.getType()->getNameWithNamespace("."));
          auto agx_str = agx::String(uuid);
          agx_obj->setUuid(agx::Uuid(agx_str));
        }
      }

      template <typename T>
      static void regenerateUuid(const openplx::Core::Object& openplx_object, T& agx_obj)
      {
        auto uuid = openplx::UuidGenerator::generateUuidv5(
          openplx_object.getUuid(), openplx_object.getType()->getNameWithNamespace("."));
        auto agx_str = agx::String(uuid);
        agx_obj.setUuid(agx::Uuid(agx_str));
      }

      template <typename T>
      static void regenerateUuid(
        const openplx::Core::Object& openplx_object, agx::ref_ptr<T>& agx_obj, std::string identifier)
      {
        if (agx_obj != nullptr) {
          auto uuid = openplx::UuidGenerator::generateUuidv5(openplx_object.getUuid(), identifier);
          auto agx_str = agx::String(uuid);
          agx_obj->setUuid(agx::Uuid(agx_str));
        }
      }

      agx::ConstraintRef lookupConstraintFromUuid(const std::string& uuid);

      agx::Line mapLine(const std::shared_ptr<openplx::Math::Line>& line) const;
      static agx::Vec3 mapVec3(const std::shared_ptr<openplx::Math::Vec3>& vec3, double scale = 1.0);
      static agx::Quat mapQuat(const std::shared_ptr<openplx::Math::Quat>& quat);
      static agx::Matrix3x3 mapMatrix3x3(const std::shared_ptr<openplx::Math::Matrix3x3>& matrix);
      static agx::AffineMatrix4x4 mapTransform(const openplx::Math::AffineTransform& transform);
      static agx::Name mapName(const std::string& name);
      agxCollide::GeometryRef handleCachedShapes(
        std::shared_ptr<openplx::Physics3D::Geometries::ContactGeometry> contact_geometry, std::string uuid);
      std::vector<agxCollide::GeometryRef> mapObjectGeometries(const std::shared_ptr<openplx::Core::Object>& object);
      void mapGeometry(
        const openplx::Physics3D::Geometries::ContactGeometry& geometry, agxCollide::GeometryRef& agx_geometry);
      agxCollide::GeometryRef mapBox(const openplx::Physics3D::Geometries::Box& box);
      agxCollide::GeometryRef mapCylinder(const openplx::Physics3D::Geometries::Cylinder& cylinder);
      agxCollide::GeometryRef mapCapsule(const openplx::Physics3D::Geometries::Capsule& capsule);
      agxCollide::GeometryRef mapConvex(const openplx::Physics3D::Geometries::ConvexMesh& convex);
      agxCollide::GeometryRef mapSphere(const openplx::Physics3D::Geometries::Sphere& sphere);
      agxCollide::GeometryRef mapExternalTriMeshGeometry(
        const openplx::Physics3D::Geometries::ExternalTriMeshGeometry& obj_geometry);
      agxCollide::GeometryRef mapTriMesh(const std::shared_ptr<openplx::Physics3D::Geometries::TriMesh>& trimesh);
      agx::RigidBodyRef mapBody(
        const std::shared_ptr<openplx::Physics3D::Bodies::RigidBody>& body, const agxSDK::AssemblyRef& assembly);
      bool inertiaTensorIsSet(const std::shared_ptr<openplx::Math::Matrix3x3>& inertia_tensor);
      /**
       * @brief map mass properties to an AGX rigid body.
       * @returns true on success, false if error occored (due to negative mass or invalid inertia tensor). Error is
       * reported to error reporter.
       *
       */
      bool mapMassProperties(
        const agx::RigidBodyRef& rigid_body, const std::shared_ptr<openplx::Physics3D::Bodies::Inertia>& inertia,
        const openplx::Math::AffineTransform& cm_transform);
      agx::MergedBodyRef mapKinematicLock(const openplx::Physics::KinematicLock& kinematic_lock);
      agxSDK::AssemblyRef mapSystem(const std::shared_ptr<openplx::Physics3D::System>& system);
      agxSDK::AssemblyRef mapSystemPass1(const std::shared_ptr<openplx::Physics3D::System>& system);
      void mapSystemPass2(
        const std::shared_ptr<openplx::Physics3D::System>& system, const agxSDK::AssemblyRef& assembly);
      void mapSystemPass3(
        const std::shared_ptr<openplx::Physics3D::System>& system, const agxSDK::AssemblyRef& assembly);
      void mapSystemPass4(
        const std::shared_ptr<openplx::Physics3D::System>& system, const agxSDK::AssemblyRef& assembly);
      void mapSystemPass5(
        const std::shared_ptr<openplx::Physics3D::System>& system, const agxSDK::AssemblyRef& assembly);

      /* Constraints */
      template <class ConstraintClass>
      agx::ref_ptr<ConstraintClass> mapInteraction(
        const std::shared_ptr<openplx::Physics::Interactions::Interaction>& interaction,
        agx::BasicControllerConstraint* controller = nullptr);
      void mapConstraintFriction(
        agx::FrictionController& agx_friction_controller,
        const std::shared_ptr<openplx::Physics::Interactions::Dissipation::DefaultFriction>& friction);
      void mapHingeFriction(
        agx::Hinge& agx_hinge, const std::shared_ptr<openplx::Physics3D::Interactions::Hinge>& hinge);
      agx::HingeRef mapHinge(const std::shared_ptr<openplx::Physics3D::Interactions::Hinge>& hinge);
      agx::SlackHingeJointRef mapSlackHinge(const std::shared_ptr<openplx::Physics3D::Interactions::Hinge>& hinge);
      void mapCylindricalFriction(
        agx::CylindricalJoint& agx_cylindrical,
        const std::shared_ptr<openplx::Physics3D::Interactions::Cylindrical>& cylindrical);
      agx::CylindricalJointRef mapCylindrical(
        const std::shared_ptr<openplx::Physics3D::Interactions::Cylindrical>& cylindrical);
                void mapEnvironment(const std::shared_ptr<openplx::Simulation::Environment>& environment);

      agx::SlackCylindricalJointRef mapSlackCylindrical(
        const std::shared_ptr<openplx::Physics3D::Interactions::Cylindrical>& cylindrical);
      void mapPrismaticFriction(
        agx::Prismatic& agx_prismatic, const openplx::Physics3D::Interactions::Prismatic& prismatic);

      agx::BallJointRef mapBall(const std::shared_ptr<openplx::Physics3D::Interactions::Ball>& ball);

      agx::PrismaticRef mapPrismatic(const std::shared_ptr<openplx::Physics3D::Interactions::Prismatic>& prismatic);

      agx::SlackPrismaticJointRef mapSlackPrismatic(
        const std::shared_ptr<openplx::Physics3D::Interactions::Prismatic>& prismatic);

      agx::LockJointRef mapLock(const std::shared_ptr<openplx::Physics3D::Interactions::Lock>& lock);

      agx::SlackLockJointRef mapSlackLock(const std::shared_ptr<openplx::Physics3D::Interactions::Lock>& lock);

      agx::DistanceJointRef mapDistance(const std::shared_ptr<openplx::Physics3D::Interactions::Distance>& distance);

      template <typename OpenPlxMotorClass, bool IS_VEL>
      void mapMotor(
        const std::shared_ptr<OpenPlxMotorClass>& motor, const agxSDK::AssemblyRef& assembly,
        agx::Angle::Type angle_type);

      static void mapRangeParameters(
        const std::shared_ptr<openplx::Physics3D::Interactions::RangeInteraction1DOF>& range,
        agx::RangeControllerRef new_range);

      void mapRange(
        const std::shared_ptr<openplx::Physics3D::Interactions::RangeInteraction1DOF>& range,
        const agxSDK::AssemblyRef& assembly, agx::Angle::Type angle_type);

      template <class OpenPlxSpringClass>
      void mapSpring(
        const std::shared_ptr<OpenPlxSpringClass>& spring, const agxSDK::AssemblyRef& assembly,
        agx::Angle::Type angle_type);

      agx::ConstraintAngleBasedData createStandaloneAngleData(agx::Angle::Type type);

      template <class OpenPlxSpringClass>
      void setSpringPositionOrAngle(agx::LockController& agx_lock, const OpenPlxSpringClass& spring);

      template <class EffortMotorClass>
      double getMotorDefaultEffort(const EffortMotorClass& motor);


      /* Tracks */
      void mapTrackAnnotations(
        const std::shared_ptr<openplx::Vehicles::Tracks::System>& track_system, agxVehicle::Track* track);
      agxVehicle::TrackRef mapTrackSystem(const std::shared_ptr<openplx::Vehicles::Tracks::System>& track_system);

      /* Contact materials*/
      agx::MaterialRef mapMaterial(const std::shared_ptr<openplx::Physics::Geometries::Material>& material);
      void mapMaterialOverride(std::shared_ptr<openplx::Core::Object> object);
      void overrideRigidBodyMaterial(std::shared_ptr<openplx::Physics3D::Bodies::RigidBody> body,
        agx::MaterialRef agx_material, std::vector<openplx::Core::Any> excluded_geometries);
      void overrideSystemMaterial(
        std::shared_ptr<openplx::Physics::System> system, agx::MaterialRef agx_material,
        std::vector<openplx::Core::Any> excluded_geometries);
      void mapSurfaceContactDefinition(
        const std::shared_ptr<openplx::Physics::Interactions::SurfaceContact::Model>& contact_model);
      agx::MaterialRef lookupMaterial(const std::shared_ptr<openplx::Physics::Geometries::Material>& material);
      void findAndMapMaterial(const std::shared_ptr<openplx::Physics::Geometries::Material>& b_material);

      /* Collision groups */
      void mapSystemToCollisionGroup(
        const std::shared_ptr<openplx::Physics3D::System>& system,
        const openplx::Simulation::CollisionGroup& collision_group);
      void mapBodyToCollisionGroup(
        const openplx::Physics3D::Bodies::Body& body, const openplx::Simulation::CollisionGroup& collision_group);
      void mapGeometryToCollisionGroup(
        const openplx::Physics3D::Geometries::ContactGeometry& geometry,
        const openplx::Simulation::CollisionGroup& collision_group);
      void mapCollisionGroup(const openplx::Simulation::CollisionGroup& collision_group);
      void disableCollisionPairs(const openplx::Core::Object& system);

      /* Tire */
      void mapWheel(
        const std::shared_ptr<openplx::Vehicles::Wheels::Base>& wheel, const agxSDK::AssemblyRef& assembly);
      agxModel::TwoBodyTireRef mapElasticWheel(
        const std::shared_ptr<openplx::Vehicles::Wheels::ElasticWheel>& wheel, const agxSDK::AssemblyRef& assembly);

      /* Robotics */
      std::list<agxModel::SuctionGripperRef> mapVacuumGripper(
        const std::shared_ptr<openplx::Robotics::EndEffectors::VacuumGripper>& b_vacuum_gripper,
        const agxSDK::AssemblyRef& assembly);
      agxModel::SuctionCupRef mapSuctionCup(
        const std::shared_ptr<openplx::Robotics::EndEffectors::SuctionCup>& suction_cup,
        agxModel::VacuumSystem* vacuum);

      /* Terrain */
      /**
       * @brief Maps an OpenPLX terrain to an AGX terrain
       * @return A new assembly containing the created AGX terrain and it's rigid body
       */
      agxSDK::AssemblyRef mapTerrain(const std::shared_ptr<openplx::Terrain::Terrain>& b_terrain);

      /**
       * @brief Adds the created AGX Shovel to all terrains
       */
      void mapShovel(const std::shared_ptr<openplx::Terrain::Shovel>& b_shovel, const agxSDK::AssemblyRef& assembly);

      /* Misc */
      agx::FrameRef mapMateConnector(const std::shared_ptr<openplx::Physics3D::Interactions::MateConnector>& mate_connector);
      agx::ObserverFrame* mapMateConnectorToObserverFrame(
        const std::shared_ptr<openplx::Physics3D::Interactions::MateConnector>& mate_connector);
      void setMotionControl(const openplx::Core::Any& any);
      void mapLocalTransform(agx::Frame& frame, const openplx::Math::AffineTransform& local_transform);
      void mapVelocities(
        agx::RigidBody* body, const std::shared_ptr<openplx::Physics3D::Bodies::RigidBody>& openplx_body);

    private:
      agxSDK::Simulation* m_simulation;
      agxPowerLine::PowerLineRef m_power_line;
      std::string m_source_id;
      openplx::Token m_root_token;
      std::shared_ptr<AgxCache> m_agx_cache;
      std::shared_ptr<AgxMetadata> m_agx_metadata;
      std::shared_ptr<OpenPlxMaterialManager> m_material_manager;
      bool m_map_drivetrain;
      bool m_map_vehicle;
      std::shared_ptr<OpenPlxSensorsMapper> m_sensors_mapper;
      std::shared_ptr<openplx::ErrorReporter> m_error_reporter;
      // Set of Mate Actuators, since the agx constraints need to exist before
      // the OpenPLX Actuator can create an agx Actuator
      // TODO: could be an Actuator, not Object, if we manage to create the base type
      std::unordered_set<std::shared_ptr<openplx::Core::Object>> m_mate_actuator_set;

      agx::Vector<agxSDK::EventListenerRef> m_event_listeners;

      // Different maps from OpenPLX instances to AGX instances
      std::unordered_map<std::shared_ptr<openplx::Physics3D::System>, agxSDK::AssemblyRef> m_system_map;
      std::unordered_map<std::shared_ptr<openplx::Core::Object>, agx::RigidBodyRef> m_rigidbody_map;
      std::unordered_map<std::shared_ptr<openplx::Core::Object>, agx::ConstraintRef> m_constraint_map;
      std::unordered_map<std::string, agx::ConstraintRef> m_agx_uuid_constraint_map;
      std::unordered_map<std::shared_ptr<openplx::Core::Object>, agxCollide::GeometryRef> m_geometry_map;
      std::unordered_map<std::shared_ptr<openplx::Core::Object>, agxPowerLine::UnitRef> m_drivetrain_unit_map;
      std::unordered_map<std::shared_ptr<openplx::Physics::Geometries::Material>, agx::MaterialRef> m_material_map;
      std::unordered_map<std::shared_ptr<openplx::Core::Object>, agx::FrameRef> m_frame_map;
      std::unordered_map<std::shared_ptr<openplx::Core::Object>, agxTerrain::TerrainRef> m_terrain_map;
      std::unordered_map<std::shared_ptr<openplx::Vehicles::Tracks::System>, agxVehicle::TrackRef> m_tracks_map;
      std::unordered_map<std::string, std::shared_ptr<openplx::Core::Object>> m_uuid_object_map;

      /* Static utility methods */
      static std::optional<double> findRelaxationTime(
        std::shared_ptr<openplx::Physics::Interactions::Dissipation::DefaultDissipation> dissipation,
        std::shared_ptr<openplx::Physics::Interactions::Flexibility::DefaultFlexibility> flexibility);
      static std::optional<double> findCompliance(
        std::shared_ptr<openplx::Physics::Interactions::Flexibility::DefaultFlexibility> flexibility);
      static std::optional<ElementaryConstraintDOF> getElementaryConstraintIndexAndRow(
        agx::Constraint& constraint, uint32_t _dof);
                std::unordered_map<std::shared_ptr<openplx::Physics3D::System>, agxSDK::AssemblyRef> m_wheel_map;

      static void setRelaxationTime(
        agx::ElementaryConstraint& e, uint32_t e_dof,
        std::shared_ptr<openplx::Physics::Interactions::Dissipation::DefaultDissipation> dissipation,
        std::shared_ptr<openplx::Physics::Interactions::Flexibility::DefaultFlexibility> flexibility);

      static void setRelaxationTime(
        agx::Constraint& constraint, uint32_t dof,
        std::shared_ptr<openplx::Physics::Interactions::Dissipation::DefaultDissipation> dissipation,
        std::shared_ptr<openplx::Physics::Interactions::Flexibility::DefaultFlexibility> flexibility);

      static void setCompliance(
        agx::ElementaryConstraint& e, uint32_t e_dof,
        std::shared_ptr<openplx::Physics::Interactions::Flexibility::DefaultFlexibility> flexibility);

      static void setCompliance(
        agx::Constraint& constraint, uint32_t dof,
        std::shared_ptr<openplx::Physics::Interactions::Flexibility::DefaultFlexibility> flexibility);

      static void setHingeRegularization(
        agx::Constraint& agx_hinge, const std::shared_ptr<openplx::Physics3D::Interactions::Hinge>& hinge);

      static void setCylindricalRegularization(
        agx::Constraint& agx_cylindrical,
        const std::shared_ptr<openplx::Physics3D::Interactions::Cylindrical>& cylindrical);

      static void setLockRegularization(agx::Constraint& agx_lock, const openplx::Physics3D::Interactions::Lock& lock);
      static void setPrismaticRegularization(
        agx::Constraint& agx_prismatic, const openplx::Physics3D::Interactions::Prismatic& prismatic);
      static void setBallRegularization(agx::Constraint& agx_ball, const openplx::Physics3D::Interactions::Ball& ball);


      friend class OpenPlxToAgxVisualsMapper;
      friend class OpenPlxDriveTrainMapper;
      friend class OpenPlxSensorsMapper;
      friend class OpenPlxVehicleMapper;
  };
}
