/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxStream/OutputArchive.h>
#include <agxStream/InputArchive.h>

#include <mutex>
#include <condition_variable>
#include <functional>
#include <type_traits>

namespace agx
{
  DOXYGEN_START_INTERNAL_BLOCK()

  template<typename FlagT>
  struct AtomicState
  {
    static_assert( std::is_enum<FlagT>::value, "Expecting enum type." );

    using Flag = FlagT;
    using ValueType = std::underlying_type_t<Flag>;
    using Flags = std::atomic<ValueType>;

    inline AtomicState( ValueType initialValue = ValueType{ 0 } )
      : m_flags{ initialValue }
    {
    }

    inline AtomicState( const AtomicState& other )
      : m_flags( other.m_flags.load() )
    {
    }

    inline AtomicState& operator= ( const AtomicState& other )
    {
      m_flags = other.m_flags.load();
      return *this;
    }

    inline AtomicState& add( Flag flag )
    {
      std::unique_lock<std::mutex> lock{ m_mutext };

      m_flags.fetch_or( (ValueType)flag );

      lock.unlock();

      m_condition.notify_one();

      return *this;
    }

    inline AtomicState& remove( Flag flag )
    {
      std::unique_lock<std::mutex> lock{ m_mutext };

      m_flags.fetch_and( (ValueType)~flag );

      lock.unlock();

      m_condition.notify_one();

      return *this;
    }

    inline AtomicState& swap( Flag fromFlag, Flag toFlag )
    {
      std::unique_lock<std::mutex> lock{ m_mutext };

      m_flags.fetch_and( (ValueType)~fromFlag );
      m_flags.fetch_or( (ValueType)toFlag );

      lock.unlock();

      m_condition.notify_one();

      return *this;
    }

    inline bool get( Flag flag ) const
    {
      return ( m_flags.load() & (ValueType)flag ) != 0;
    }

    inline AtomicState& waitAndSet( Flag waitForFlag, Flag setFlag )
    {
      return waitWhile( waitForFlag, [this, setFlag]() { set( setFlag ); } );
    }

    inline AtomicState& waitWhile( Flag flagIsSet )
    {
      return waitWhile( flagIsSet, {} );
    }

    inline AtomicState& waitWhile( Flag flagIsSet, std::function<void()>&& onDone )
    {
      std::unique_lock<std::mutex> lock{ m_mutext };
      m_condition.wait( lock,
                        [this, flagIsSet]()
                        {
                          return !get( flagIsSet );
                        } );

      if ( onDone )
        onDone();

      lock.unlock();

      m_condition.notify_one();

      return *this;
    }

    inline void store( agxStream::OutputArchive& out ) const
    {
      out << agxStream::out( "state", (ValueType)m_flags.load() );
    }

    inline void restore( agxStream::InputArchive& in )
    {
      ValueType state{};
      in >> agxStream::in( "state", state );
      m_flags = state;
    }

    private:
      inline AtomicState& set( Flag flag )
      {
        m_flags.fetch_or( (ValueType)flag );

        return *this;
      }

    private:
      std::mutex m_mutext;
      std::condition_variable m_condition;
      Flags m_flags;
  };

  DOXYGEN_END_INTERNAL_BLOCK()
}
