
#pragma once

#include <memory>
#include <vector>
#include <unordered_set>
#include <openplx/ErrorReporter.h>
#include <openplx/ModelDeclaration.h>
#include <openplx/SortedAssignment.h>

namespace openplx {
    /**
     * Implements [Kahn's algorithm](https://www.youtube.com/watch?v=cIBFEhD77b4) to produce a sorted topological order of assignments.
     *
     */
    class TopologicalSort {
        public:

            /**
             * Iterates all OuterSymbols in model_declaration and const_models
             * and filters out the ones that are relevant for evaluating model_declaration,
             * in principle all annotations and attributes with values. We also include
             * attributes that are "empty" models. This means we filter out symbols
             * without any assignments and also method declaration symbols.
             */
            static std::unordered_set<OuterSymbolPtr> findRelevantSymbols(
                    const ModelDeclPtr& model_declaration,
                    const std::vector<ModelDeclPtr>& const_models);

            /**
             * Runs kahns algorithm using the GetSortedNodesVisitor and UpdateDegreeVisitor, stores the result in the model declaration.
             *
             * @param model_declaration Input/Output: The model declaration that is the scope of the sort, the result can be extracted using getTopologicalsort() on model_declaration.
             * @param const_models Input: The const model declarations that is in scope.
             * @param error_reporter Output: The error reporter. Should be checked by caller!
             */
            static std::vector<OuterSymbolPtr> sortOuterSymbolTree(
                const ModelDeclPtr& model_declaration,
                const std::vector<ModelDeclPtr>& const_models,
                const std::shared_ptr<ErrorReporter>& error_reporter);

            /**
             * Adds edges from each symbol to it's nested symbols, i.e. before we can
             * evaluate foo.x we must evaluate foo (if foo has a value of it's own)
             */
            static void addEdgesDownstream(
                std::shared_ptr<TreeNode<OuterSymbol>> branch,
                std::deque<OuterSymbolPtr>& stack,
                std::unordered_set<OuterSymbolPtr>& candidates);

            /**
             * Find and tag symbols that are references, used for error handling
             */
            static void tagReferences(
                const ModelDeclPtr& model,
                const std::vector<ModelDeclPtr>& const_models);

            /**
             * Check for NestedDeclarationsNotAllowed, MissingAssignment and ReferencesAreReadOnly errors
             */
            static void validateSymbols(
                const ModelDeclPtr& model,
                const std::vector<ModelDeclPtr>& const_models,
                const std::shared_ptr<ErrorReporter>& error_reporter);

            /**
             * Used by validateSymbols
             */
            static void validateReadOnly(
                const std::shared_ptr<TreeNode<OuterSymbol>>& branch,
                bool read_only,
                bool no_nested,
                size_t no_nested_depth,
                const std::shared_ptr<ErrorReporter>& error_reporter
            );
    };
}
