#pragma once

#include <memory>
#include <openplx/Node.h>
#include <openplx/Type.h>

namespace openplx {
    /**
     * The OuterSymbol represents the second intermediate representation in the
     * semantic analysis. Using the TreeNode class we can build a symbol tree
     * of OuterSymbol nodes that resolves which syntax tree nodes have the
     * outer most precedence for each symbol.
     */
    class OuterSymbol : public Type, public std::enable_shared_from_this<OuterSymbol>
    {
        public:
            static std::shared_ptr<OuterSymbol> create();

            std::string toKey() override;

            bool isAssignableTo(const TypePtr& other) override;

            /* Overrides */
            void unbind() override;
            OuterSymbolPtr asOuterSymbol() override;
            void accept(NodeVisitor& visitor) override;
            TypePtr resolve() override;

            /* Accessors */
            TypePtr getType() const;
            TypePtr getPreviousType() const;
            TypePtr getOriginalType() const;
            ExprPtr getValue() const;
            ExprPtr getBoundValue() const;
            size_t getThisOffset() const;
            size_t getIsOffset() const;
            bool isReference() const;
            bool hasReferenceAncestor() const;

            ModelDeclPtr getRootModel() const;
            NodePtr getSourceOfType() const;
            NodePtr getSourceOfValue() const;
            NodePtr getSourceOfPreviousType() const;
            NodePtr getSourceOfOriginalType() const;
            const std::vector<ModelDeclPtr>& getTraits() const;
            bool hasTrait(ModelDeclPtr trait) const;

            /* Mutations */
            void setRootModel(ModelDeclPtr root_model);
            void setSourceOfType(NodePtr source_of_type);
            void setSourceOfValue(NodePtr source_of_value);
            void setSourceOfPreviousType(NodePtr source_of_previous_type);
            void setSourceOfOriginalType(NodePtr source_of_original_type);
            void setThisOffset(size_t this_offset);
            void setIsOffset(size_t is_offset);
            void setBoundValue(ExprPtr value);
            void setHasReferenceAncestor(bool has_reference_ancestor);
            void appendToTraits(ModelDeclPtr trait);

            /* Path */
            const std::vector<std::string>& getPath() const;
            void appendToPath(std::string segment);

            /* Merge operations */
            void mergeLeft(OuterSymbol& other);
            void mergeRight(OuterSymbol& other);
            OuterSymbolPtr clone() const;

            std::vector<std::string> calculateThisPath() const;
            std::string toString() override;

        protected:
            OuterSymbol();
            ModelDeclPtr m_root_model;
            NodePtr m_source_of_type;
            NodePtr m_source_of_value;
            NodePtr m_source_of_previous_type;
            NodePtr m_source_of_original_type;
            std::vector<ModelDeclPtr> m_traits;
            ExprPtr m_bound_value;
            size_t m_this_offset;
            size_t m_is_offset;
            bool m_has_reference_ancestor;
            std::vector<std::string> m_path;
    };
}
