#pragma once

#include <memory>
#include <string>
#include <filesystem>
#include <optional>
#include <unordered_map>
#include <unordered_set>
#include <openplx/OpenPlxContext.h>
#include <openplx/BundleConfig.h>
#include <openplx/AnalysisContext.h>
#include <openplx/ErrorReporter.h>
#include <openplx/BundleLookup.h>
#include <openplx/EvaluatorContext.h>
#include <openplx/DocumentContext.h>
#include <openplx/Plugin.h>
#include "openplx/RuntimeContext.h"

namespace openplx::Core::Api
{
    class OpenPlxContextInternal
    {
        public:

            explicit OpenPlxContextInternal(std::vector<std::string> paths);
            ~OpenPlxContextInternal();

            static OpenPlxContextInternal* fromContext(OpenPlxContext& openplx_context)
            {
                return openplx_context.m_internal.get();
            }

            /**
             * @brief Parse string and all bundles add them to the context.
             */
            std::vector<DocPtr> parseString(const std::string& openplx_text);

            /**
             * @brief Parse openplx_file and bundles and add them to the context.
             * If the openplx_file is part of a bundle, only the bundle dependencies are parsed, otherwise all bundles are parsed.
             */
            std::vector<DocPtr> parseFile(const std::filesystem::path& openplx_file);

            /**
             * @brief Parse openplx_file with virtual content and bundles and add them to the context.
             * If the openplx_file is part of a bundle, only the bundle dependencies are parsed, otherwise all bundles are parsed.
             */
            std::vector<DocPtr> parseFileWithVirtualContent(const std::filesystem::path& openplx_file, const std::string& content);

            /**
             * @brief Parse all bundles and add them and the document to the context
             */
            DocPtr parseBundlesWithDocument(const DocPtr& document);

            std::optional<Bundle::BundleConfig> parseBundle(const std::filesystem::path& openplx_config_file);
            std::vector<Bundle::BundleConfig> parseAllBundles(std::vector<std::string> extra_paths);

            void prepareAnalysis();
            std::vector<std::shared_ptr<DocumentContext>> documents();
            std::shared_ptr<openplx::ErrorReporter> getErrorReporter() const;
            std::shared_ptr<openplx::Analysis::AnalysisContext> analysisContext();
            std::shared_ptr<openplx::Core::EvaluatorContext> evaluatorContext();
            const RuntimeContext& getRuntimeContext() const;
            void registerPlugin(std::shared_ptr<Plugin> plugin);
            std::vector<std::shared_ptr<openplx::Plugin>> getRegisteredPlugins();
            std::vector<Bundle::BundleConfig> findBundleConfigs(std::vector<std::string> extra_paths) const;
            std::vector<Bundle::BundleConfig> getBundleConfigs() const;

            void release();

            void clear();

            void useUuidv5(std::string namespace_id);

            const std::unordered_set<openplx::Core::ObjectPtr>& getRegisteredObjects() const;

            void enableCache();
            void disableCache();
            bool cacheEnabled() const;
            void invalidateCache(std::string source_id);

        private:
            void parseBundles();
            void syncDocumentContexts(Bundle::BundleConfig& config);
            std::optional<Bundle::BundleConfig> loadOrCreateConfigForFile(const std::filesystem::path& openplx_file);
            std::unordered_set<std::string> resolvePluginExtensions();

            std::unordered_map<std::string, std::shared_ptr<openplx::Plugin>> m_plugins;
            std::vector<std::string> m_bundle_paths;
            std::unordered_multimap<std::string, std::shared_ptr<DocumentContext>> m_document_contexts;
            std::shared_ptr<openplx::Analysis::AnalysisContext> m_analysis_context;
            std::shared_ptr<openplx::Core::EvaluatorContext> m_eval_ctx;

            std::unique_ptr<openplx::RuntimeContext> m_runtime_context;

            bool m_cache_enabled;
    };
}
