#pragma once

#include <memory>
#include <string>
#include <sstream>
#include <vector>
#include <unordered_map>
#include <openplx/Token.h>
#include <openplx/NodeVisitor.h>
#include <openplx/Types.h>

namespace openplx {
    /**
     * The Formatter takes a syntax tree document and converts it to a string, respecting the line and
     * column data present in the tokens to the extent where it is possible (some information
     * is lost when converting a text document to a syntax tree).
     *
     * Comments are preserved but will be reformatted so that they are placed two spaces
     * from the previous token on the line. Or if the comment is the only token on that line
     * it will retain its formatting.
     */
    class Formatter : private NodeVisitor {
        public:
            Formatter();
            std::string format(const DocPtr& document);
            std::vector<Token> collectTokens(const DocPtr& document);
            static std::string formatTokens(const std::vector<Token>& tokens);
            static void fixOverlappingTokens(std::vector<Token>& tokens);

            void skipComments();
            void skipMethods();
            void skipOperatorOverloads();
            void skipImports();
            void setUseImportOriginalPathToken(bool use_import_original_path_token);

        private:

            static std::string tokenToString(const Token& token);

            void visitDocument(const DocPtr& document) override;

            void visitModelDeclaration(const ModelDeclPtr& model_declaration) override;

            void visitVarAssignment(const VarAssignPtr& var_assignment) override;

            void visitMethodDeclaration(const MethodDeclPtr& method_declaration) override;

            void visitConstant(const ConstantPtr& constant) override;

            void visitCall(const CallPtr& call) override;

            void visitMemberAccess(const MemberAccessPtr& member_access) override;

            void visitArray(const ArrayPtr& array) override;

            void visitUnary(const UnaryPtr& unary) override;

            void visitBinaryOp(const BinaryOpPtr& binary_op) override;

            void visitTraitImpl(const TraitImplPtr& trait_impl) override;

            void visitParameter(const ParamPtr& parameter) override;

            void visitIndexing(const IndexingPtr& indexing) override;

            void visitOperatorOverload(const OpOverloadPtr& op_overload) override;

            void visitAnnotation(const AnnotationPtr& annotation) override;

            void visitImport(const ImportPtr& ptr) override;

            void visitInitializer(const InitializerPtr& initializer) override;

            void visitDeletion(const DeletionPtr& del) override;

        std::vector<Token> m_tokens;
        bool m_skip_comments;
        bool m_skip_methods;
        bool m_skip_operator_overloads;
        bool m_skip_imports;
        bool m_use_import_original_path_token;
    };
}
