/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxTerrain/export.h>
#include <agx/agx_vector_types.h>
#include <agx/Prismatic.h>
#include <agxCollide/Box.h>
#include <agx/Line.h>

namespace agxTerrain
{
  class Shovel;
  class Terrain;
  class TerrainMaterial;

  AGX_DECLARE_POINTER_TYPES(SoilPenetrationResistance);
  class AGXTERRAIN_EXPORT SoilPenetrationResistance : public agx::Referenced
  {
  public:

    enum class PenetrationModel
    {
      NO_MODEL,
      ELASTIC_PLASTIC_LIMIT,
      PLASTIC
    };

    SoilPenetrationResistance();

    SoilPenetrationResistance(Shovel* shovel);

    /**
    \internal

    On addNotification of Terrain and Shovel.
    */
    void addNotification(agxSDK::Simulation* simulation);

    /**
    \internal

    On removeNotification of Terrain and Shovel.
    */
    void removeNotification(agxSDK::Simulation* simulation);

    /**
    \internal

    On pre step of Terrain and Shovel.
    */
    void onPre(Terrain* terrain, Shovel* shovel);

    /**
    \internal

    On pre step of Terrain and Shovel.
    */
    void onPreCollide( Shovel* shovel );

    PenetrationModel getPenetrationModel();

    void setPenetrationModel(PenetrationModel model);

    agx::Prismatic* getPenetrationPrismatic();

    void setupPenetrationResistance(Terrain* terrain, Shovel* shovel);

    void updatePrismaticFrames( Shovel* shovel );

    agx::Vec3 getPrismaticDirectionWorld() const;

    bool isActive();

    void onEnableChange( bool enable );

    void updateTeethGeometry( Shovel* shovel );

    void updateTeethPlacement( agxTerrain::Shovel* shovel );

    agxCollide::Geometry* getTeethGeometry() const;

    /**
    \internal

    Set whether the penetration resistance should be active or not by setting the enable flag on the
    penetration prismatic motor.
    \param enable - set to true/false if the penetration resistance should be active or not
    */
    void setActive(bool enable);

  protected:
    virtual ~SoilPenetrationResistance();

  private:

    void _updateTeethGeometry( agx::Real toothLength,
                               agx::Real toothWidth,
                               agx::Real toothDiameter );

    void updateTeethGeometryTransform( const agx::Vec3& position,
                                       const agx::Quat& rot );

    agx::Real calculateToothPressureElasticPlasticLimit(agx::Real cohesion,
                                                        agx::Real poissionRatio,
                                                        agx::Real frictionAngle,
                                                        agx::Real E,
                                                        agx::Real p0,
                                                        agx::Real a,
                                                        agx::Real a0);

    agx::Real calculateToothPressurePlastic(agx::Real cohesion,
                                            agx::Real poissionRatio,
                                            agx::Real frictionAngle,
                                            agx::Real E,
                                            agx::Real p0,
                                            agx::Real dilatancyAngle,
                                            agx::Real a,
                                            agx::Real a0);

    void calculateToothRadiusAndDepth(Terrain* terrain, Shovel* shovel, agx::Real& a, agx::Real& h);

    agx::Real calculateSeparatingPlateArea(Terrain* terrain, Shovel* shovel, agx::Vec3Vector& contactPositions);

    agx::Real calculateSeparatingPlateMassParticles(Terrain* terrain, Shovel* shovel);

    void getPressureAndYoungsModulusAtTeeth( Terrain* terrain,
                                             Shovel* shovel,
                                             agx::UInt nTeeth,
                                             agx::Real& p0,
                                             agx::Real& E,
                                             agx::Real& compaction,
                                             agx::Real& meanDilatancyAngle,
                                             agx::Real& meanDepth,
                                             agx::Real swell );

    agx::Vec3iVector voxelIndicesAlongEdge(Terrain* terrain, agx::Line edge);

    agx::Real calculateSeparatingPlateForce(Terrain* terrain, Shovel* shovel);

    agx::Real calculateToothForce(Terrain* terrain, Shovel* shovel);

    void calculatePowerSeriesCoefficients(const agx::Real gamma,
                                          const agx::Real xi,
                                          const agx::UInt n,
                                          agx::RealVector& coefficients);

    agx::Real calculatePowerSeries(const agx::Real x,
                                   const agx::Real gamma,
                                   const agx::UInt n,
                                   const agx::RealVector& coefficients);

    agx::Real calculatePressureLimit(const agx::Real m,
                                     const agx::Real beta,
                                     const agx::Real gamma,
                                     const agx::Real delta,
                                     const agx::Real eta,
                                     const agx::UInt n,
                                     const agx::RealVector& coefficients);

    agx::Real calculatePressureRatio(const agx::Real R_limit,
                                     const agx::Real a,
                                     const agx::Real a0,
                                     const agx::Real m,
                                     const agx::Real beta,
                                     const agx::Real gamma,
                                     const agx::Real delta,
                                     const agx::Real eta,
                                     const agx::UInt n,
                                     const agx::RealVector& coefficients);

  private:
    agx::PrismaticRef       m_penetrationPrismatic;
    PenetrationModel        m_model;
    agxCollide::GeometryRef m_teethGeometry;
    agxCollide::BoxRef      m_teethBox;
    bool             m_enable;
  };
}