/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/BitState.h>
#include <agxTerrain/export.h>
#include <agx/Line.h>
#include <agx/agx_vector_types.h>
#include <agxStream/OutputArchive.h>
#include <agxStream/InputArchive.h>
#include <agx/Referenced.h>


namespace agxTerrain
{

  class AdvancedShovelSettings;
  AGX_DECLARE_POINTER_TYPES(AdvancedShovelSettings);


  AGX_DECLARE_POINTER_TYPES(ShovelSettings);
  class AGXTERRAIN_EXPORT ShovelSettings : public agx::Referenced
  {
  public:

    /*
    Settings class for shovel.
    */
    ShovelSettings(const agx::Line& cuttingEdge, const agx::Vec3& toothDirection, agx::Real toothLength);

    // Common settings

    /**
    Set true/false if the shovel deformers should make particle free deformations. (Default: false)
    Note, if this is true all excavation modes will make particle free deformations.
    Even if enableCreateDynamicMass is set to false for one or more excavation modes.
    \param enable - true if the shovel deformers mode should make particle free deformations.
    */
    void setEnableParticleFreeDeformers(bool enable);
    /**
    \return whether the shovel uses particle free deformations.
    */
    bool getEnableParticleFreeDeformers() const;

    /**
    Set the vertical penetration depth threshold for when the shovel tooth for penetration
    resistance should reach full effectiveness. The penetration depth is defined as the vertical
    distance between the tip of a shovel tooth and the surface position of the height field. The
    penetration resistance will increase from a baseline of 10% until maximum effectiveness is reached
    when the vertical penetration depth of the shovel reaches the specified value. (Default: 0.5 m)
    \param depthThreshold - the vertical pressure threshold that will cause penetration
                            resistance to start.
    */
    void setPenetrationDepthThreshold(agx::Real depthThreshold);
    /**
    Get the vertical penetration depth threshold for when the shovel tooth for penetration
    resistance should reach full effectiveness. The penetration depth is defined as the vertical
    distance between the tip of a shovel tooth and the surface position of the height field. The
    penetration resistance will increase from a baseline of 10% until maximum effectiveness is reached
    when the vertical penetration depth of the shovel reaches the specified value. (Default: 0.5 m)
    \return the vertical depth penetration threshold that will cause penetration resistance to
            reach full effectiveness.
    */
    agx::Real getPenetrationDepthThreshold() const;

    /**
    Set the linear scaling coefficient for the penetration force (Default: 1.0)
    \param penetrationForceScaling - The coefficient for scaling the penetration force that the terrain will give on this shovel
    */
    void setPenetrationForceScaling(agx::Real penetrationForceScaling);

    /**
    \return the linear scaling coefficient for the penetration force (Default: 1.0)
    */
    agx::Real getPenetrationForceScaling() const;


    /**
    Set a maximum limit on penetration force (N) that the terrain will generate on this shovel. (Default: Infinity)
    \param maxPenetrationForce - The maximum penetration force that the terrain will act on this shovel
    */
    void setMaxPenetrationForce(agx::Real maxPenetrationForce);
    /**
    Set the maximum limit on penetration force (N) that the terrain will generate on this shovel. (Default: Infinity)
    \return The maximum penetration force that the terrain will act on this shovel
    */
    agx::Real getMaxPenetrationForce() const;

    /**
    \param length - new tooth length. Default: 0.15 m.
    */
    void setToothLength(agx::Real length);

    /**
    \returns the tooth length of the shovel. Default: 0.15 m.
    */
    agx::Real getToothLength() const;

    /**
    \returns the tooth direction of the shovel. Default: [0, 0, 0].
    */
    agx::Vec3 getToothDirection() const;

    /**
    Set the tooth direction of the shovel. This will affect the
    direction along which the penetration resistance will act on
    the shovel body when submerged into the terrain. It will also
    affect the direction of the generated teeth geometry if it is
    enabled.

    \param toothDirection - The direction of the teeth. Default: [0, 0, 0].
    */
    void setToothDirection( const agx::Vec3& toothDirection);

    /**
    Set whenever custom tooth geometry dimensions should be used
    instead of automatically generating a teeth geometry from the
    shovel settings. Also see: setCustomTeethGeometryHalfVec.
    \param enable - True if custom geometry dimensions should be used,
                    false otherwise. Default: False.
    */
    void setEnableCustomTeethGeometry( bool enable );

    /**
    Get whenever custom tooth geometry dimensions should be used
    instead of automatically generating a teeth geometry from the
    shovel settings. Also see: setCustomTeethGeometryHalfVec.
    \return true if custom geometry dimensions should be used,
            false otherwise. Default: False.
    */
    bool getEnableCustomTeethGeometry() const;

    /**
    Set the custom tooth geometry box half vector dimensions to
    be used when generating a custom teeth geometry, if it is enabled.
    @see setEnableCustomTeethGeometry
    \param halfVec - The dimensions of the custom teeth geometry box
                     to be used if custom teeth geometry is enabled.
    */
    void setCustomTeethGeometryHalfVec( const agx::Vec3& halfVec );

    /**
    Get the custom tooth geometry box half vector dimensions to
    be used when generating a custom teeth geometry, if it is enabled.
    @see setEnableCustomTeethGeometry
    \return the dimensions of the custom teeth geometry box
            to be used if custom teeth geometry is enabled.
    */
    agx::Vec3 getCustomTeethGeometryHalfVec() const;

    /**
    \param radius - new minimum radius of the shovel teeth. Default: 0.015 m.
    */
    void setToothMinimumRadius(agx::Real radius);

    /**
    \returns the minimum radius of the shovel teeth. Default: 0.015 m.
    */
    agx::Real getToothMinimumRadius() const;

    /**
    \param radius - new maximum radius of the shovel teeth. Default: 0.075 m.
    */
    void setToothMaximumRadius(agx::Real radius);

    /**
    \returns the maximum radius of the shovel teeth. Default: 0.075 m.
    */
    agx::Real getToothMaximumRadius() const;

    /**
    \param count - new teeth count of shovel. Default: calculated as ceil((cuttingEdge.p2 - cuttingEdge.p1).length() / 0.5).
    */
    void setNumberOfTeeth(agx::UInt count);
    /**
    \returns the teeth count of the shovel evenly distributed along the cutting edge. Default: calculated as ceil((cuttingEdge.p2 - cuttingEdge.p1).length() / 0.5).
    */
    agx::UInt getNumberOfTeeth() const;

    /**
    Set whenever a teeth geometry should be generated between the
    cutting edge and the teeth edge from the teeth radius and length.
    \param enable - true if a teeth geometry should be generated, false otherwise.
    */
    void setEnableGenerateTeethGeometry( bool enable );

    /**
    \returns true if teeth geometry should be generated for the shovel, false otherwise.
    */
    bool getEnableGenerateTeethGeometry() const;

    /**
    Set whenever a excavation at the teeth edge should be enabled so
    that digging starts at the teeth edge instead of at the cutting edge.
    \param enable - true if excavation at the teeth edge should be enabled, false otherwise.
    \note - When excavation at teeth edge is disabled then teeth geometry generation is also disabled.
    */
    void setEnableExcavationAtTeethEdge( bool enable );

    /**
    \returns true if excavation at the teeth edge should be enabled so
             that digging starts at the teeth edge instead of at the cutting edge, false otherwise.
    */
    bool getEnableExcavationAtTeethEdge() const;

    /**
    Sets the vertical distance under the blade cutting edge that the soil is
    allowed to instantly merge up to. Default: 0.0 m.
    */
    void setVerticalBladeSoilMergeDistance(agx::Real verticalSoilBladeMergeDistance);
    /**
    Get the vertical distance under the blade cutting edge that the soil is
    allowed to instantly merge up to. Default: 0.0 m.
    */
    agx::Real getVerticalBladeSoilMergeDistance() const;

    DOXYGEN_START_INTERNAL_BLOCK()

    /*
    Constructor for serialization
    */
    ShovelSettings();

    ~ShovelSettings();

    void store(agxStream::OutputArchive& out) const;
    void restore( agxStream::InputArchive& in);

    /* This method is only used internally for restore from TerrainToolCollection */
    void restorePreShovelRefactor(agxStream::InputArchive& in, const agx::Line& cuttingEdge, bool& enabled);

    /**
    \return advanced settings for the shovel
    */
    AdvancedShovelSettings* getAdvancedSettings() { return m_advanced; }
    AdvancedShovelSettings* getAdvancedSettings() const { return m_advanced; }

    bool isDirty() const { return m_dirty; };
    void setDirty( bool enable ) { m_dirty = enable; }

    DOXYGEN_END_INTERNAL_BLOCK()

  private:

    enum CommonFlags : agx::UInt32
    {
      ENABLE_PARTICLE_FREE_DEFORMATION = 1 << 0,
      ENABLE_GENERATE_TEETH_GEOMETRY = 1 << 1,
      ENABLE_CUSTOM_TEETH_GEOMETRY = 1 << 3,
      ENABLE_EXCAVATION_AT_TEETH_EDGE = 1 << 4
    };
    using CommonBitState = agx::BitState<CommonFlags, agx::UInt32>;

    CommonBitState           m_flags;

    /* teeth parameters for the penetration resistance model */
    struct TeethProperties
    {
      agx::Real   toothLength;
      agx::Real   toothMinRadius;
      agx::Real   toothMaxRadius;
      agx::UInt   numberOfTeeth;
      agx::Vec3   toothDirection;
    };
    TeethProperties             m_teethProps;
    agx::Vec3                   m_customTeethGeometryHalfVec;

    /* struct for soil penetration parameters */
    struct SoilPenetrationParameters
    {
      agx::Real depthThreshold;
      agx::Real forceScaling;
      agx::Real maxForce;
    };

    SoilPenetrationParameters m_soilPenetrationParameters;
    agx::Real                 m_verticalSoilBladeMergeDistance;
    // Contains advanced settings for the shovel
    AdvancedShovelSettingsRef m_advanced;

    bool m_dirty;
  };

  /*
  Advanced settings class for shovel.
  */
  class AGXTERRAIN_EXPORT AdvancedShovelSettings : public agx::Referenced
  {
  public:

    /**
    Constructor of advanced shovel settings from the necessary initial info from a shovel.
    */
    AdvancedShovelSettings(const agx::Line& cuttingEdge);

    /**
    Constructor for serialization
    */
    AdvancedShovelSettings();

    /**
    Set to true/false if shovel <-> terrain contacts should always be removed. Default: false.
    \param enable - set to true/false if shovel <-> terrain contacts should always be removed
    */
    void setAlwaysRemoveShovelContacts(bool enable);

    /**
    \return true/false if shovel <-> terrain contacts are removed. Default: false.
    */
    bool getAlwaysRemoveShovelContacts() const;

    /**
    Set if inner shape alone should always create dynamic mass. The alternative is to only
    create dynamic mass in the inner shape when primary excavation soil wedges create mass.
    Default: true.
    \param enable - true if inner shape should always create dynamic mass (default), false otherwise.
    */
    void setEnableInnerShapeCreateDynamicMass(bool enable);

    /**
    Get if inner shape alone should always create dynamic mass. The alternative is to only
    create dynamic mass in the inner shape when primary excavation soil wedges create mass. Default: true.
    \return true if inner shape should always create dynamic mass (default), false otherwise.
    */
    bool getEnableInnerShapeCreateDynamicMass() const;

    /**
    Set whenever the excavation force feedback during PRIMARY excavation should be
    generated from particle contacts instead of aggregate contacts.
    \param enable - true if particles should generate contact
                    forces on the shovel, false otherwise. (Default: false)
    */
    void setEnableParticleForceFeedback(bool enable);

    /**
    \return whenever the excavation force feedback during PRIMARY excavation should be
    generated from particle contacts instead of aggregate contacts. (Default: false)
    */
    bool getEnableParticleForceFeedback();

    /**
    Set whenever the innerbody aggregate should have zero velocity before solve step as
    opposed to using particle or cached post-solve velocities from the previous time step.
    \param enable - true if the innerbody aggregate should have zero velocity, false otherwise. Default: false.
    */
    void setUseZeroAggregateVelocity(bool enable);

    /**
    \return true if the innerbody aggregate should have zero velocity, false otherwise. Default: false.
    */
    bool getUseZeroAggregateVelocity() const;

    /*
    Set whenever custom friction models should be used in the
    shovel <-> aggregate contacts.
    \param enable - True if custom friction models should be used. (Default: true)
    */
    void setUseCustomFrictionModel(bool enable);

    /**
    \return whenever custom friction models should be used in the shovel <-> aggregate contacts. (Default: true)
    */
    bool getUseCustomFrictionModel() const;

    /**
    Set the extension outside the shovel bounding box where soil particle merging is forbidden. Default: 0.5 m.
    */
    void setNoMergeExtensionDistance(agx::Real extensionDistance);

    /**
    Get the margin outside the shovel bonding box where soil particle merging is forbidden. Default: 0.5 m.
    */
    agx::Real getNoMergeExtensionDistance() const;

    /**
    Set the minimum submerged cutting edge length fraction [0-1] that generates submerged cutting. Default: 0.5
    */
    void setMinimumSubmergedContactLengthFraction(agx::Real minimumSubmergedContactLengthFraction);

    /**
    Get the minimum submerged cutting edge length fraction [0-1] that generates submerged cutting. Default: 0.5
    */
    agx::Real getMinimumSubmergedContactLengthFraction() const;

    /**
    Sets the dead-load limit where secondary separation will start to activate where the forward direction
    starts to change according to the virtual separation plate created by the material inside the shovel
    \param secondarySeparationLimit - The dead load limit where secondary separation will start to activate (Default: 0.8)
    */
    void setSecondarySeparationDeadloadLimit(agx::Real secondarySeparationLimit);

    /**
    \return the dead-load limit where secondary separation will start to active where the forward direction
            starts to change according to the virtual separation plate created by the material inside the shovel (Default: 0.8)
    */
    agx::Real getSecondarySeparationDeadloadLimit() const;

    /**
    Set the starting distance threshold from the shovel planes where regular geometry contacts between
    the shovel underside and the terrain can be created. Contacts that are not past the distance threshold
    will be filtered away.
    \param contactRegionThreshold - The contact distance threshold from the shovel planes where regular
                                    geometry contacts between the shovel underside and the terrain are
                                    allowed to be created. ( Default: cuttingEdgeLength / 10.0 )
    */
    void setContactRegionThreshold(agx::Real contactRegionThreshold);

    /**
    Get the starting distance threshold from the shovel planes where regular geometry contacts between
    the shovel underside and the terrain can be created. Contacts that do not reach the distance threshold
    will be filtered away.
    \return the starting distance threshold from the shovel planes where regular geometry contacts between
            the shovel underside and the terrain can be created. ( Default: cuttingEdgeLength / 10.0 )
    */
    agx::Real getContactRegionThreshold() const;

    /**
    Set the maximum vertical distance from the shovel bottom plane where regular geometry contacts between
    the shovel and the terrain are allowed to be created. Contacts past the distance will be filtered away.
    \param verticalLimit - The vertical contact distance threshold from the shovel planes where
                                      regular geometry contacts between the shovel underside and the terrain are
                                      allowed to be created. ( Default: cuttingEdgeLength / 10.0 )
    */
    void setContactRegionVerticalLimit(agx::Real verticalLimit);

    /*
    Get the maximum vertical distance from the shovel bottom plane where regular geometry
    contacts between the shovel and the terrain are allowed to be created. Contacts past that distance
    will be filtered away.
    \return the maximum vertical distance from the shovel bottom plane where regular geometry
            contacts between the shovel and the terrain are allowed to be created. ( Default: cuttingEdgeLength / 10.0 )
    */
    agx::Real getContactRegionVerticalLimit() const;

    /**
    Set the radius multiplier for extending the inclusion bound with
    particle radius during post-excavation with particles in bucket.
    \param radiusMultiplier - The multiplier for the particle radius
                              extension of the shovel inclusion
                              bound. (Default: 1.0)
    \note - This will only be active pos-excavation and NOT during excavation
            when we have active soil wedges.
    */
    void setParticleInclusionMultiplier(agx::Real radiusMultiplier);

    /**
    \return the radius multiplier for extending the inclusion bound with
    particle radius during post-excavation with particles in bucket. (Default: 1.0)
    \note - This will only be active post-excavation and NOT during excavation
            when we have active soil wedges.
    */
    agx::Real getParticleInclusionMultiplier() const;

    /**
    Set the custom excavation stiffness multiplier, which is used to multiply the Young's modulus of the user defined
    contact materials in the different excavation modes. See Shovel::getShovelTerrainContactMaterialContainer().
    \param excavationStiffnessMultiplier - the excavation stiffness multiplier for the explicit contact materials. Default: 1.0.
    */
    void setCustomExcavationStiffnessMultiplier(agx::Real excavationStiffnessMultiplier);

    /**
    \return the custom excavation stiffness multiplier, which is used to multiply the Young's modulus of the user defined
    contact materials in the different excavation modes. See Shovel::getShovelTerrainContactMaterialContainer(). Default: 1.0.
    */
    agx::Real getCustomExcavationStiffnessMultiplier() const;

    /**
    Set number of contact patches that will be created inside
    the shovel to generate shovel <-> aggregate contact points.
    \param numContactPatches - the number of contact patches that will be generated.
    */
    void setNumContactPatches( agx::UInt numContactPatches );

    /**
    Get the number of contact patches that will be created inside
    the shovel to generate shovel <-> aggregate contact points.
    */
    agx::UInt getNumContactPatches() const;

    /**
    Set whenever custom contact partitions should be used or not.
    \note - see setCustomContactPatchAngleTresholds.
    */
    void setEnableCustomPartitioning( bool enable );

    /**
    \return whenever custom contact partitions should be used or not.
    \note - see setCustomContactPatchAngleTresholds.
    */
    agx::Bool getEnableCustomPartitioning() const;

    /**
    Specify cutsom contact patches by givin custom angle thresholds.
    Will be used if custom partitioning is enabled.
    \note - see setEnableCustomPartitioning.
    */
    void setCustomContactPatchAngleTresholds( const agx::RealVector& angleThresholds );

    /**
    Specify cutsom contact patches by givin custom angle thresholds.
    Will be used if custom partitioning is enabled.
    \note - see setEnableCustomPartitioning.
    */
    agx::RealVector getCustomContactPatchAngleThresholds() const;

    /**
    Set the number of sampling points used when tracing the shovel outline
    during creation of contact patches.
    */
    void setNumContactPatchSamplingPoints( agx::UInt numSamples );

    /**
    \return the number of sampling points used when tracing the shovel outline
    during creation of contact patches.
    */
    agx::UInt getNumContactPatchSamplingPoints() const;

    /**
    Set the angle cutoff from the cutting edge when the contact sampling
    inside the shovel will stop.
    */
    void setContactSamplingAngleCutoff( agx::Real angleCutoff );

    /**
    \return the angle cutoff from the cutting edge when the contact sampling
    inside the shovel will stop.
    */
    agx::Real getContactSamplingAngleCutoff() const;

    /**
    Set if the inner shape aggregate velocity should explicitly be
    set to the shovel velocity.
    */
    void setAggregateVelocityToShovel( bool enable );

    /**
    \return if the inner shape aggregate velocity should explicitly be
    set to the shovel velocity.
    */
    agx::Bool getAggregateVelocityToShovel() const;

    /**
    Set if shovel <-> aggregate contacts should be set to use
    non-holonomic constraints, i.e velocity constraints.
    */
    void setNonHolonomicContacts( bool enable );

    /**
    \return if shovel <-> aggregate contacts should be set to use
    non-holonomic constraints, i.e velocity constraints.
    */
    agx::Bool getNonHolonomicContacts() const;

    /**
    Set if the inner shape aggregate velocity should clamped to the
    total soil particle velocity.
    */
    void setClampAggregateVelocityToParticles( bool enable );

    /**
    \return if the inner shape aggregate velocity should clamped to the
    total soil particle velocity.
    */
    agx::Bool getClampAggregateVelocityToParticles() const;

    /**
    Set if we should disabled wedge aggregates for primary excavation.
    \note - this setting is used by the ClamShellBucket.
    */
    void setDisablePrimaryWedgeAggregates( bool enable );

    /**
    \return if we should disabled wedge aggregates for primary excavation
    \note - this setting is used by the ClamShellBucket.
    */
    agx::Bool getDisablePrimaryWedgeAggregates() const;

    /**
    Set if we limit the particle inner shape inclusion to the particle
    midpoint.
    */
    void setEnableInnerShapeMidpointInclusion( bool enable );

    /**
    \return if we limit the particle inner shape inclusion to the particle
    midpoint.
    */
    agx::Bool getEnableInnerShapeMidpointInclusion() const;

    /**
    Set if shovel <-> aggregate side contacts should be disabled.
    */
    void setDisableShovelSideContacts( bool enable );

    /**
    \return if shovel <-> aggregate side contacts should be disabled.
    */
    bool getDisableShovelSideContacts() const;

    /**
    Set if we should update contact patches of the shovel, this is set
    when the shovel contact patch settings are changed.
    \note - not serialized.
    */
    void setShouldUpdateContactPatches( bool shouldUpdate );

    /**
    \return Set if we should update contact patches of the shovel, this is set
    when the shovel contact patch settings are changed.
    \note - not serialized.
    */
    bool getShouldUpdateContactPatches() const;

    /**
    Enables or disables the use of a custom forward vector.

    When enabled, the simulation will use the custom forward vector
    set with setCustomForwardVector() instead of the default one
    generated from the shovel geometry. This can be useful
    if the geometric algorithms fails on a particular shovel shape.

    \param enable - true to enable the custom forward vector, false to disable it.
    */
    void setEnableCustomForwardVector( bool enable );

    /**
    Checks whether a custom forward vector is enabled.

    \return true if a custom forward vector is enabled, false otherwise.
    */
    bool getEnableCustomForwardVector() const;

    /**
    Enables or disables the use of a custom bottom vector.

    When enabled, the simulation will use the custom bottom vector
    set with setCustomBottomVector() instead of the default one
    generated from the shovel geometry.

    \param enable - true to enable the custom bottom vector, false to disable it.
    */
    void setEnableCustomBottomVector( bool enable );

    /**
    Checks whether a custom bottom vector is enabled.

    \return true if a custom bottom vector is enabled, false otherwise.
    */
    bool getEnableCustomBottomVector() const;

    /**
    Sets the custom forward vector.

    When enabled the custom forward vector will replace the one
    that is generated from the shovel geometry. This can be useful
    if the geometric algorithms fails on a particular shovel shape.

    \param forwardVector - The vector to use as the custom forward direction.
    @see setEnableCustomForwardVector()
    */
    void setCustomForwardVector( const agx::Vec3& forwardVector );

    /**
    \return the current custom forward vector.
    */
    agx::Vec3 getCustomForwardVector() const;

    /**
    Sets the custom bottom plate direction vector.

    When enabled the custom bottom plate direction vector will replace the one
    that is generated from the shovel geometry. This can be useful
    if the geometric algorithms fails on a particular shovel shape.

    \param bottomVector The vector to use as the custom bottom direction.
    @see setEnableCustomBottomVector()
    */
    void setCustomBottomVector( const agx::Vec3& bottomVector );

    /**
    \return the current custom bottom vector.
    */
    agx::Vec3 getCustomBottomVector() const;

    /**
    Enables or disables the teeth volume feature.

    When enabled, a volumetric representation of the space between
    the top edge, cutting edge and the teeth edge is generated and
    used for including particles in the shovel inner shape.

    \param enable True to enable teeth volume, false to disable it.
    */
    void setEnableTeethVolume( bool enable );

    /**
    Checks whether the teeth volume feature is enabled.

    \return True if the teeth volume is enabled, false otherwise.
    */
    bool getEnableTeethVolume() const;

    bool isDirty() const { return m_dirty; };

    void setDirty( bool enable ) { m_dirty = enable; }

  protected:
    friend class ShovelSettings;
    void store(agxStream::OutputArchive& out) const;
    void restore(agxStream::InputArchive& in);

  private:
    enum AdvancedFlags : agx::UInt32
    {
      REMOVE_SHOVEL_CONTACTS = 1 << 0,
      INNER_SHAPE_CREATE_DYNAMIC_MASS = 1 << 1,
      ENABLE_PARTICLE_FREE_DEFORMATION = 1 << 2,
      ENABLE_PARTICLE_FORCE_FEEDBACK = 1 << 3,
      SET_ZERO_AGGREGATE_VELOCITY = 1 << 4,
      USE_CUSTOM_FRICTION_MODELS = 1 << 5,
      CLAMP_AGGREGATE_VELOCITY_TO_PARTICLES = 1 << 6,
      SET_AGGREGATE_VELOCITY_TO_SHOVEL = 1 << 7,
      SET_NON_HOLONOMIC_CONTACTS = 1 << 8,
      CUSTOM_PATCH_PARTITIONING = 1 << 9,
      DISABLE_PRIMARY_WEDGE_AGGREGATES = 1 << 10,
      INNER_SHAPE_STRICT_PARTICLE_INCLUSION = 1 << 11,
      DISABLE_SHOVEL_SIDE_CONTACTS = 1 << 12,
      ENABLE_CUSTOM_FORWARD_VECTOR = 1 << 13,
      ENABLE_CUSTOM_BOTTOM_VECTOR = 1 << 14,
      DISABLE_TEETH_SHAPE = 1 << 15,
    };
    using AdvancedBitState = agx::BitState<AdvancedFlags, agx::UInt32>;
    AdvancedBitState          m_advancedFlags;

    agx::Real                 m_noMergeExtensionDistance;
    agx::Real                 m_minimumSubmergedContactLengthFraction;
    agx::Real                 m_secondarySeparationDeadLoadLimit;
    agx::Real                 m_particleInclusionMultiplier;
    agx::Real                 m_contactThreshold;
    agx::Real                 m_verticalContactLimit;
    agx::Real                 m_excavationStiffnessMultiplier;

    agx::UInt                 m_numContactPatches;
    agx::UInt                 m_numPatchSamplingPoints;
    agx::Real                 m_contactSamplingAngleCutoff;
    agx::RealVector           m_customAngleThresholds;
    bool                      m_shouldUpdateContactPatches;
    agx::Vec3                 m_customForwardVector;
    agx::Vec3                 m_customBottomVector;
    bool                      m_dirty;
  };

  DOXYGEN_START_INTERNAL_BLOCK()

  AGX_DECLARE_POINTER_TYPES( ShovelDebugRenderSettings );
  class AGXTERRAIN_EXPORT ShovelDebugRenderSettings : public agx::Referenced
  {
  public:
    ShovelDebugRenderSettings();

    void setRenderForbiddenBounds( bool enable );
    bool getEnableRenderForbiddenBounds() const;

    void setEnableRenderAggregateContactArea( bool enable );
    bool getEnableRenderAggregateContactArea() const;

    void setEnableRenderAggregateContacts( bool enable );
    bool getEnableRenderAggregateContacts() const;

    void setEnableRenderContactPatches( bool enable );
    bool getEnableRenderContactPatches() const;

    void setEnableRenderInnerParticles( bool enable );
    bool getEnableRenderInnerParticles() const;

    void setEnableRenderWedgeParticles( bool enable );
    bool getEnableRenderWedgeParticles() const;

    void setEnableRenderWedgeVoxels( bool enable );
    bool getEnableRenderWedgeVoxels() const;

    void setEnableRenderExcludedInnerParticles( bool enable );
    bool getEnableRenderExcludedInnerParticles() const;

    void setEnableRenderContactPartitioning( bool enable );
    bool getEnableRenderContactPartitioning() const;

    void setEnableRenderSideWallContactPartitioning( bool enable );
    bool getEnableRenderSideWallContactPartitioning() const;

    void setRenderWedgeLocks( bool enable );
    bool getRenderWedgeLocks() const;

    void setParticleScalingFactor( agx::Real scalingFactor );
    agx::Real getParticleScalingFactor() const;

    void setRenderActiveZone( bool enable );
    bool getRenderActiveZone() const;

    void setRenderSoilWedges( bool enable );
    bool getRenderSoilWedges() const;

    void setRenderInnerShapeAlgorithm( bool enable );
    bool getRenderInnerShapeAlgorithm() const;

    void setRenderContactPatchIntersectionTest( bool enable );
    bool getRenderContactPatchIntersectionTest() const;

    void setContactRenderSize( agx::Real size );
    agx::Real getContactRenderSize() const;

    void setRenderTeeth( bool enable );
    bool getRenderTeeth() const;

    void setRenderShovelPrimaryVectors( bool enable );
    bool getRenderShovelPrimaryVectors() const;

    void setRenderShovelDeformerVectors( bool enable );
    bool getRenderShovelDeformerVectors() const;

    void setRenderInnerShapes( bool enable );
    bool getRenderInnerShapes() const;

    void setRenderBottomPlateAlgorithm( bool enable );
    bool getRenderBottomPlateAlgorithm() const;

    void setRenderInnerPlaneAlgorithm( bool enable );
    bool getRenderInnerPlaneAlgorithm() const;

    void setRenderShovelDeadloadParticles( bool enable );
    bool getRenderShovelDeadloadParticles() const;

    void setRenderSecondarySeparation( bool enable );
    bool getRenderSecondarySeparation() const;

  private:
    enum AdvancedFlags : agx::UInt32
    {
      RENDER_FORBIDDEN_BOUNDS = 1 << 0,
      RENDER_AGGREGATE_CONTACTS = 1 << 1,
      RENDER_AGGREGATE_CONTACT_AREA = 1 << 2,
      RENDER_AGGREGATE_CONTACT_PARTITIONING = 1 << 3,
      RENDER_AGGREGATE_SIDE_CONTACT_PARTITIONING = 1 << 4,
      RENDER_INNER_SHAPE_PARTICLES = 1 << 5,
      RENDER_WEDGE_SHAPE_PARTICLES = 1 << 6,
      RENDER_EXCLUDED_INNER_SHAPE_PARTICLES = 1 << 7,
      RENDER_CONTACT_PATCHES = 1 << 8,
      RENDER_WEDGE_LOCKS = 1 << 9,
      RENDER_SOIL_WEDGES = 1 << 10,
      RENDER_ACTIVE_ZONE = 1 << 11,
      RENDER_INNER_SHAPE_ALGORITHM = 1 << 12,
      RENDER_CONTACT_PATCH_INTERSECTION_TEST = 1 << 13,
      RENDER_SHOVEL_TEETH = 1 << 14,
      RENDER_SHOVEL_PRIMARY_VECTORS = 1 << 15,
      RENDER_SHOVEL_DEFORMER_VECTORS = 1 << 16,
      RENDER_SHOVEL_INNER_SHAPES = 1 << 17,
      RENDER_BOTTOM_PLANE_ALGORITHM = 1 << 18,
      RENDER_INNER_PLANE_ALGORITHM = 1 << 19,
      RENDER_SHOVEL_DEADLOAD_PARTICLES = 1 << 20,
      RENDER_SHOVEL_SECONDARY_SEPARATION = 1 << 21,
      RENDER_WEDGE_VOXELS = 1 << 22
    };
    using RenderState = agx::BitState<AdvancedFlags, agx::UInt32>;
    RenderState         m_flags;
    agx::Real           m_particleScalingFactor;
    agx::Real           m_contactRenderSize;
  };

  DOXYGEN_END_INTERNAL_BLOCK()
}
