/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agx/Referenced.h>
#include <agx/FrictionModel.h>
#include <agx/agx_vector_types.h>
#include <agx/BitState.h>
#include <agxTerrain/GroundCollapseUtils.h>
#include <agxStream/Serializable.h>


namespace agxTerrain
{
  namespace GroundCollapse
  {
    AGX_DECLARE_POINTER_TYPES(CollapseSettings);

    /*
    Class that manages settings for the ground collapse algorithm.
    */
    class AGXTERRAIN_EXPORT CollapseSettings : public agx::Referenced, public agxStream::Serializable
    {
    public:
      CollapseSettings();

      /*
      Set the number of points on the lower side of the slope. This affects how many failure surfaces are tested during the ground collapse calculations.
      \param numAPoints - the desired number of points on the lower side of the slope. Default: 1.
      */
      inline void setNumLowerPoints( agx::UInt numLowerPoints ) { m_numLowerPoints = numLowerPoints; };
      /*
      Get the number of points on the lower side of the slope. Default: 1.
      */
      inline agx::UInt getNumLowerPoints() { return m_numLowerPoints; };


      /*
      Set the number of points on the upper side of the slope. This affects how many failure surfaces are tested during the ground collapse calculations.
      \param numBPoints - the desired number of points on the upper side of the slope. Default: 1.
      */
      inline void setNumUpperPoints( agx::UInt numUpperPoints ) { m_numUpperPoints = numUpperPoints; };
      /*
      Get the number of points on the upper side of the slope. Default: 1.
      */
      inline agx::UInt getNumUpperPoints() { return m_numUpperPoints; };

      /*
      Set the number of varied radii for the constructed failure surfaces. Default: 1.
      \param radiusNumber - the desired number of varying radii.
      */
      inline void setRadiusNumber( agx::UInt radiusNumber ) { m_numberOfRadii = radiusNumber; };
      /*
      Get the number of varied radii for the constructed failure surfaces. Default: 1.
      */
      inline agx::UInt getRadiusNumber() { return m_numberOfRadii; };

      /*
      Set the contact point configuration of the created wedges.
      \param pointConfigiration - the contact point configuration of the created wedges. Default: [3, 3, 3].
      */
      void setPointConfiguration( const GroundCollapse::PointConfiguration& pointConfiguration );

      /*
      Set the simulation frequency by the ground collapse simulation.
      \param simulationFrequency - the simulation frequency that should be used by the ground collapse simulation. Default: 60 Hz.
      */
      void setSimulationFrequency(agx::Real simulationFrequency);

      /*
      Get the contact point configuration of the created wedges. Default: [3, 3, 3].
      */
      GroundCollapse::PointConfiguration getPointConfiguration() const;

      /*
      Set the shrink factor used during contact generation between the soil aggregate and the terrain. Default: [0.125, 0.125].
      \param contactShrinkFactor - the shrink factor to use during contact generation.
      */
      void setContactShrinkFactor( const agx::Vec2& contactShrinkFactor );

      /*
      \return the shrink factor used during contact generation. Default: [0.125, 0.125].
      */
      agx::Vec2 getContactShrinkFactor() const;

      /*
      Set the fraction of the upper point span which offsets the zone in the reverse 
      slope direction, to ensure all contacts are included.
      \param offsetFraction - the fraction to offset the zone by. Default: 0.1.
      */
      void setUpperPointSpanOffset( agx::Real offsetFraction );

      /*
      \return the fraction of the upper point span which offsets the zone in the reverse 
      slope direction, to ensure all contacts are included. Default: 0.1.
      */
      agx::Real getUpperPointSpanOffset() const;

      /*
      Set the offset fraction from the slope start where the lower points on the slope will start to be generated from.
      The lower points will be generated from the starting point on the slope calculated as: start = \p offset * ( slopesUpperEndPoint - slopesLowerEndPoint)
      \param offset - the desired offset for the generation of the slope lower points. Default: 0.25.
      */
      void setLowerPointSpanOffset( agx::Real offset );

      /*
      Get the offset from the slope start where the lower points on the slope will start to be generated from.
      The lower points will be generated from the starting point on the slope calculated as: start = \p offset * ( slopesUpperEndPoint - slopesLowerEndPoint)
      \return the lower points offset. Default: 0.25.
      */
      agx::Real getLowerPointSpanOffset();

      /*
      Set the scale factor to modify the length span where the lower points will be located on the slope.
      \param scaleFactor - the fraction to increase the slope length. Default: 1.25.
      */
      void setLowerPointSpanScaleFactor( agx::Real scaleFactor );

      /*
      \return the scale factor of the length span where the lower points will be located on the slope. Default: 1.25.
      */
      agx::Real getLowerPointSpanScaleFactor() const;

      /*
      Set the number of intersection points used to determine the intersection of the
      failure surface in the terrain when constructing the aggregate contact system.
      \param numIntersectionPoints - the number of intersection points to use in the failure surface <-> terrain intersection test. Default: 100.
      */
      inline void setNumIntersectionPoints( agx::UInt numIntersectionPoints ) { m_numIntersectionPoints = numIntersectionPoints; };

      /*
      Get the number of intersection points used to determine the intersection of the
      failure surface in the terrain when constructing the aggregate contact system. Default: 100.
      */
      inline agx::UInt getNumIntersectionPoints() { return m_numIntersectionPoints; };

      /*
      Set the acceleration threshold for failure surfaces.
      \param threshold - acceleration threshold to use. Default: 0.1.
      */
      void setLinearAccelerationThreshold( agx::Real threshold );

      /*
      \return the acceleration threshold used for failure surface. Default: 0.1.
      */
      agx::Real getLinearAccelerationThreshold() const;

      /*
      Set the contact viscosity for the soil aggregate contacts.
      \param viscosity- viscosity to use in aggregate contacts. Default: 1E-12.
      */
      void setAggregateContactViscosity( agx::Real viscosity );

      /*
      Set the solve type for the soil aggregate contacts.
      \param setSolveType - enum value signifying which solveType should be used in the aggregate contacts. Default: agx::FrictionModel::SolveType::ITERATIVE.
      */
      void setSolveType( agx::FrictionModel::SolveType setSolveType );

      /*
      \return the contact viscosity for the soil aggregate contacts. Default: 1E-12.
      */
      agx::Real getAggregateContactViscosity() const;

      /*
      Set whether the algorithm should stop after it finds a failure surface that fails.
      Note: Only used for serial ground collapse simulation!
      \param shouldStop - if the ground collapse simulation should stop or continue after
      it has found a failed failure surface. Default: false.
      */
      void setShouldStopAfterFirstFailure( bool shouldStop );

      /*
      Note: Only used for serial ground collapse simulation!
      \return if the ground collapse simulation stops after the first failure surface is found. Default: false.
      */
      bool getShouldStopAfterFirstFailure() const;

      /*
      Set whether the critical failure surface should collapse into dynamic mass or not.
      \param enable - true for enabling ground collapse to dynamic mass, false for disabled. Default: false.
      */
      void setEnableDynamicCollapse( bool enable );

      /*
      \return true if ground collapse will be turned to dynamic mass, false if not. Default: false.
      */
      bool getEnableDynamicCollapse() const;

      /*
      Enable whether results should be saved for all substeps.
      \param enable - true for enabling saving substep results. Default: false.
      */
      void setEnableSubStepResults(bool enable);

      /*
      Get whether results should are saved for all substeps or not. Default: false.
      */
      bool getEnableSubStepResults() const;

      /*
      \return the number of simulation substeps used by the ground collapse simulation. Default: 1.
      */
      inline agx::UInt getNumSimulationSubsteps() { return m_numSimulationSubSteps; };

      /*
      Set the number of simulation substeps used by the ground collapse simulation.
      \param numSimulationSubsteps - how many substeps the ground collapse simulation will use. Default: 1.
      */
      inline void setNumSimulationSubsteps( agx::UInt numSimulationSubSteps ) {  m_numSimulationSubSteps = numSimulationSubSteps; };

      /*
      \return the freuqency used by the ground collapse simulation. Default: 60 Hz.
      */
      inline agx::Real getSimulationFrequency() { return m_simulationFrequency; };

      /*
      \return which friction model is used by the ground collapse simulation.  Default: agx::FrictionModel::SolveType::ITERATIVE.
      */
      agx::FrictionModel::SolveType getSolveType() const;

      /*
      Set whether a custom contact material should be used between soil aggregate <-> terrain
      in the ground collapse simulation. Set the custom contact material with
      `setCustomContactMaterial`. Default: false.
      \param enable - true for enabling custom contact material, false otherwise.
      */
      void setUseCustomContactMaterial( bool enable );

      /*
      \return true if a custom contact material should be used between soil
      aggregate <-> terrain in the ground collapse simulation. False otherwise. Default: false.
      */
      bool getUseCustomContactMaterial() const;

      /*
      Set whether parallel solve of the test failure surfaces should be enabled/disabled. If enabled,
      all failure surfaces will be created and solved in parallel. If disabled, they will be serially
      constructed and solved.
      \param enable - true for enabling parallel solve. False for disable. Default: true.
      */
      void setEnableParallelSolve( bool enable );

      /*
      \return true if parallel solve is enabled, false if disabled. Default: true.
      */
      bool getEnableParallelSolve() const;

      /*
      Set whether custom user supplied failure surfaces should be used in the ground collapse simulation.
      \param enable - true if the custom failure surfaces should be used. Default: false.
      */
      void setUseCustomFailureSurfaces( bool enable );

      /*
      \return true if custom user supplied failure surfaces are used in the ground collapse simulation.
      False if they are not used. Default: false.
      */
      bool getUseCustomFailureSurfaces() const;

      /*
      Sets if the slope contacts should be represented as holonomic constraints.
      \param enable - true if the contacts should be holonomic constraints, false otherwise. Default: False
      */
      void setUseHolonomicContacts( bool enable );

      /*
      \return true if the contacts should be holonomic constraints, false otherwise. Default: False
      */
      bool getUseHolonomicContacts() const;

      /*
      Set a custom contact material to use between soil aggregate <-> terrain in the ground collapse simulation.
      To be used, must also be enabled via `setUseCustomContactMaterial`.
      \param cm - the custom contact material.
      */
      void setCustomContactMaterial( agx::ContactMaterial* cm );

      /*
      \return the custom contact material which is used if `getUseCustomContactMaterial` is true.
      */
      agx::ContactMaterial* getCustomContactMaterial() const;

      enum StateFlags : agx::UInt32
      {
        USE_CUSTOM_CONTACT_MATERIAL = 1 << 0,
        STOP_AFTER_FIRST_FAILURE = 1 << 1,
        SHOULD_COLLAPSE_SOIL = 1 << 2,
        SHOULD_SAVE_SUBSTEP_RESULTS = 1 << 3,
        ENABLE_PARALLEL_SOLVE = 1 << 4,
        USE_CUSTOM_SURFACES = 1 << 5,
        USE_HOLONOMIC_CONTACTS= 1 << 6
      };
      using Flags = agx::BitState<StateFlags, agx::UInt32>;
    public:
      AGXSTREAM_DECLARE_SERIALIZABLE(agxTerrain::GroundCollapse::CollapseSettings);

    private:
      Flags     m_flags;
      agx::UInt m_numLowerPoints;
      agx::UInt m_numUpperPoints;
      agx::UInt m_numberOfRadii;
      agx::Real m_lowerPointSpanScaleFactor;
      agx::Real m_searchZoneOffsetFraction;
      agx::Real m_lowerPointsOffset;

      agx::Real m_simulationFrequency;
      agx::UInt m_numSimulationSubSteps;

      agx::UInt m_numIntersectionPoints;
      agx::Vec2 m_contactShrinkFactor;
      PointConfiguration m_pointConfiguration;
      agx::Real m_aggregateLinearAccelerationThreshold;
      agx::Real m_aggregateContactSurfaceViscosity;
      agx::FrictionModel::SolveType m_solveType;
      agx::ContactMaterialRef m_customMaterial;
    };

    DOXYGEN_START_INTERNAL_BLOCK()

    AGX_DECLARE_POINTER_TYPES(SlopeDetectionSettings);
    /*
    Class for handling settings for the slope detection algorithm. The slope detection works so that it creates an
    average mid point of all the terrain contacts inside the ground collapse zone, and then projects a number of
    rays in a 360- degree sweep around the mid point. The rays are then discretized by a number of points along each ray.
    Each ray is then examined to see if they contain a slope.

    The settings can configure the number of rays around the mid point, the length of the rays from the mid point, and how many
    points are discretized along the rays.
    */
    class AGXTERRAIN_EXPORT SlopeDetectionSettings : public agx::Referenced, public agxStream::Serializable
    {
    public:
      SlopeDetectionSettings();

      /*
      Set the length of each ray in the slope detection algorithm from the force weighted mid point in the collapse zone.
      \param lineLength - the lenght of the rays. Default: 25.
      */
      inline void setSlopeLineLength(agx::Real lineLength) { m_slopeLineLength = lineLength; };

      /*
      Get the length of the rays used in the slope detection algorithm. Default: 25.
      */
      inline agx::Real getSlopeLineLength() { return m_slopeLineLength; };

      /*
      Set the number of points used to discretize each ray in the slope detection algorithm.
      \param numSlopePoints - the number of points along each ray. Default: 100.
      */
      inline void setNumSlopePoints(agx::UInt numSlopePoints) { m_numSlopePoints = numSlopePoints; };
      /*
      Get the number of points used to discretize each ray in the slope detection algorithm. Default: 100.
      */
      inline agx::UInt getNumSlopePoints() { return m_numSlopePoints; };

      /*
      Set the number of rays used to project outwards from the force weighted mid point in the collapse zone.
      \param numSlopeRays - the number of rays used in the slope dectection algorithm. Default: 40.
      */
      inline void setNumSlopeRays(agx::UInt numSlopeRays) { m_numSlopeRays = numSlopeRays; };

      /*
      Get the number of rays used to project outwards from the force weighted mid point in the collapse zone. Default: 40.
      */
      inline agx::UInt getNumSlopeRays() { return m_numSlopeRays; };

      AGXSTREAM_DECLARE_SERIALIZABLE(agxTerrain::GroundCollapse::SlopeDetectionSettings)
      
    private:
      agx::Real m_slopeLineLength;
      agx::UInt m_numSlopePoints;
      agx::UInt m_numSlopeRays;
    };
    DOXYGEN_END_INTERNAL_BLOCK()


    AGX_DECLARE_POINTER_TYPES( CollapseRenderSettings );
    /*
    Class that handles render settings for the slope detection and ground collapse algorithm.
    */
    class AGXTERRAIN_EXPORT CollapseRenderSettings : public agx::Referenced, public agxStream::Serializable
    {
    public:
      CollapseRenderSettings();

      /*
      Enable or disable debug graphics related to slope detection. Default: false.
      */
      void setEnableRenderSlopeDetection( bool enable ) { m_flags.update( StateFlags::RENDER_SLOPE_DETECTION, enable ); }

      /*
      \return if debug graphics related to slope detection are enabled or disabled. Default: false.
      */
      bool getEnableRenderSlopeDetection() const { return m_flags.Is( StateFlags::RENDER_SLOPE_DETECTION ); }

      /*
      Enable or disable debug graphics related to steepest slope data. Default: false.
      */
      void setEnableRenderSteepestSlopeData( bool enable ) { m_flags.update( StateFlags::RENDER_STEEPEST_SLOPE_DATA, enable ); }

      /*
      \return if debug graphics related to steepest slope data are enabled or disabled. Default: false.
      */
      bool getEnableRenderSteepestSlopeData() const { return m_flags.Is( StateFlags::RENDER_STEEPEST_SLOPE_DATA ); }

      /*
      Enable or disable debug graphics of the divided segments on the test failure surfaces. Default: false.
      */
      void setEnableRenderSurfaceIntersectionTest( bool enable ) { m_flags.update( StateFlags::RENDER_SURFACE_TERRAIN_INTERSECTION_TEST, enable ); }

      /*
      \return if debug graphics of the divided segments on the test failure surfaces is enabled or disabled. Default: false.
      */
      bool getEnableRenderSurfaceIntersectionTest() const { return m_flags.Is( StateFlags::RENDER_SURFACE_TERRAIN_INTERSECTION_TEST ); }

      /*
      Enable or disable debug graphics of the generated contacts between the failure surface <-> terrain. Default: false.
      */
      void setEnableRenderAllSurfaceContacts( bool enable ) { m_flags.update( StateFlags::RENDER_ALL_SURFACE_CONTACTS, enable ); }

      /*
      \return if debug graphics of the generated contacts between the failure surface <-> terrain are enabled or disabled. Default: false.
      */
      bool getEnableRenderAllSurfaceContacts() const { return m_flags.Is( StateFlags::RENDER_ALL_SURFACE_CONTACTS ); }

      /*
      \return if debug graphics of the generated contacts of failed failure surfaces are enabled or disabled. Default: false.
      */
      bool getEnableRenderFailedSurfaceContacts() const { return m_flags.Is( StateFlags::RENDER_FAILED_SURFACE_CONTACTS ); }

      /*
      Enable or disable debug graphics of the generated contacts in failed failure surfaces. Default: false.
      */
      void setEnableRenderFailedSurfaceContacts( bool enable ) { m_flags.update( StateFlags::RENDER_FAILED_SURFACE_CONTACTS, enable ); }

      /**
      Enable or disable if contacts should be colored according to their factor of saftey. Default: false.
      */
      void setEnableColorContactsFromFos( bool enable ) { m_flags.update( StateFlags::COLOR_CONTACTS_FROM_FOS, enable ); }

      /**
      \return if contacts should be colored according to their factor of saftey. Default: false.
      */
      bool getEnableColorContactsFromFos() const { return m_flags.Is( StateFlags::COLOR_CONTACTS_FROM_FOS ); }

      /*
      Set the renderable size for the surface contacts. Default: 0.05.
      */
      void setRenderContactSize( agx::Real size ) { m_renderContactSize = size; }

      /*
      Get the renderable size for the surface contacts. Default: 0.05.
      */
      agx::Real getRenderContactSize() const { return m_renderContactSize; }

      /*
      Enable or disable debug graphics of failed failure surfaces. Default: false.
      */
      void setEnableRenderCriticalSurface( bool enable ) { m_flags.update( StateFlags::RENDER_FAILED_SURFACES, enable ); }

      /*
      \return if debug graphics of failed failure surfaces is enabled or disabled. Default: false.
      */
      bool getEnableRenderCriticalSurface() const { return m_flags.Is( StateFlags::RENDER_FAILED_SURFACES ); }

      /*
      Enable or disable debug graphics of failed failure surfaces. Default: false.
      */
      void setEnableRenderFailedSurfaces( bool enable ) { m_flags.update( StateFlags::RENDER_FAILED_SURFACES, enable ); }

      /*
      \return if debug graphics of failed failure surfaces is enabled or disabled. Default: false.
      */
      bool getEnableRenderFailedSurfaces() const { return m_flags.Is( StateFlags::RENDER_FAILED_SURFACES ); }

      /*
      Enable or disable debug graphics of all failure surfaces. Default: false.
      */
      void setEnableRenderAllSurfaces( bool enable ) { m_flags.update( StateFlags::RENDER_ALL_SURFACES, enable ); }

      /*
      \return if the debug graphics of all failure surfaces is enabled or disabled. Default: false.
      */
      bool getEnableRenderAllSurfaces() const { return m_flags.Is( StateFlags::RENDER_ALL_SURFACES ); }

      /*
      Enable or disable debug graphics of points along the failure surface. Default: false.
      */
      void setEnableRenderSurfacePoints( bool enable ) { m_flags.update( StateFlags::RENDER_SURFACE_POINTS, enable ); }

      /*
      return if deubug graphics of points along the failure surface is enabled or disabled. Default: false.
      */
      bool getEnableRenderSurfacePoints() const { return m_flags.Is( StateFlags::RENDER_SURFACE_POINTS ); }

      /*
      Enable or disable debug graphics of the contact partitioning. Default: false.
      */
      void setEnableRenderContactPartitioning( bool enable ) { m_flags.update( StateFlags::RENDER_CONTACT_PARTITIONING, enable ); }

      /*
      \return if debug graphics of the contact partitioning is enabled or disabled. Default: false.
      */
      bool getEnableRenderContactPartitioning() const { return m_flags.Is( StateFlags::RENDER_CONTACT_PARTITIONING ); }

      /*
      Enable or disable debug graphics aggregate mass voxels. Default: false.
      */
      void setEnableRenderAggregateMassVoxels( bool enable ) { m_flags.update( StateFlags::RENDER_AGGREGATE_MASS_VOXELS, enable ); }

      /*
      Enable or disable debug graphics aggregate mass voxels. Default: false.
      */
      bool getEnableRenderAggregateMassVoxels() { return m_flags.Is( StateFlags::RENDER_AGGREGATE_MASS_VOXELS ); }

      /*
      Enable or disable debug graphics aggregate mass voxels. Default: false.
      */
      void setEnableRenderAggregateBorderVoxels( bool enable ) { m_flags.update( StateFlags::RENDER_AGGREGATE_BORDER_VOXELS, enable ); }

      /*
      Enable or disable debug graphics aggregate mass voxels. Default: false.
      */
      bool getEnableRenderAggregateBorderVoxels() { return m_flags.Is( StateFlags::RENDER_AGGREGATE_BORDER_VOXELS ); }

      enum StateFlags : agx::UInt32
      {
        RENDER_SLOPE_DETECTION = 1 << 0,
        RENDER_STEEPEST_SLOPE_DATA = 1 << 1,
        RENDER_STEEPEST_SLOPE = 1 << 2,
        RENDER_SLOPE_START_END_POINT = 1 << 3,
        RENDER_SURFACE_TERRAIN_INTERSECTION_TEST = 1 << 4,
        RENDER_ALL_SURFACE_CONTACTS = 1 << 5,
        RENDER_FAILED_SURFACE_CONTACTS = 1 << 6,
        COLOR_CONTACTS_FROM_FOS = 1 << 7,
        RENDER_ALL_SURFACES = 1 << 8,
        RENDER_FAILED_SURFACES = 1 << 9,
        RENDER_CRITICAL_SURFACE = 1 << 10,
        RENDER_SURFACE_POINTS = 1 << 11,
        RENDER_CONTACT_PARTITIONING = 1 << 12,
        RENDER_AGGREGATE_MASS_VOXELS = 1 << 13,
        RENDER_AGGREGATE_BORDER_VOXELS = 1 << 14
      };
      using Flags = agx::BitState<StateFlags, agx::UInt32>;
      
      DOXYGEN_START_INTERNAL_BLOCK()
      AGXSTREAM_DECLARE_SERIALIZABLE(agxTerrain::GroundCollapse::CollapseRenderSettings)
      DOXYGEN_END_INTERNAL_BLOCK()
    protected:
      ~CollapseRenderSettings();

    protected:
      Flags m_flags;
      agx::Real m_renderContactSize;
    };
  }
}
