/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/SystemNode.h>

namespace agxSensor
{
  struct RtNode;
  class RtSystemNodeProxy;

  AGX_DECLARE_POINTER_TYPES( RtSystemNode );

  /**
  A system node focused on raytracing. It's responsible of connecting nodes (RtNode*)
  to its parents and children. If the nature of this node is to not have a RtNode != nullptr
  from getOrCreateNode(), then it's important to override onChildAdded, (and/or) onChildRemoved,
  (and/or) onParentConnected (and/)or onParentDissconnect to manage the update system
  tree. If this node cannot manage these conditions a warning should be issued.
  */
  class AGXSENSOR_EXPORT RtSystemNode : public SystemNode
  {
    public:
      /**
      \return node (RtNode*) that has been created, otherwise nullptr
      */
      RtNode* getNode() const;

      /**
      Called when the node is required. If this method returns nullptr it's important
      that the system update connections are handled by onChildAdded, (and/or) onChildRemoved,
      (and/or) onParentConnected (and/)or onParentDissconnect.
      \return RtNode instance if this node, nullptr if this node can handle the parent
              change and added children.
      */
      RtNode* getOrCreateNode();

      template<typename T>
      RtSystemNode* findChildProxyFor() const;

    public:
      virtual ~RtSystemNode() = default;

      /**
      Called when getOrCreateNode() returns nullptr when this node is being
      added as child to \p parent.
      \param parent - parent this node is being added to
      */
      virtual void onParentConnect( RtSystemNode& parent );

      /**
      Called when getNode() returns nullptr when this node is being removed
      from its parent.
      \param parent - parent this node is being removed from
      */
      virtual void onParentDisconnect( RtSystemNode& parent );

      virtual bool addNotification() override;
      virtual void removeNotification() override;

      virtual void onChildAdded( SystemNode& child ) override;
      virtual void onChildRemoved( SystemNode& child ) override;
      virtual void cleanup() override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSensor::RtSystemNode );

    protected:
      virtual RtNode* createNode() = 0;

      virtual void store( agxStream::OutputArchive& out ) const override;
      virtual void restore( agxStream::InputArchive& in ) override;

      DOXYGEN_END_INTERNAL_BLOCK()

    protected:
      RtNode* m_node{ nullptr };
  };
}
