/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/export.h>

#include <agx/HashFunction.h>

namespace agxSensor
{
  /**
  Entity id > 0 to access user data given raytrace results including ENTITY_ID_I32.
  Note that the underlying type, e.g., int32_t, is there for compatibility reasons
  with an external library. Do not change.

  Entity id == 0 is invalid, null entity.
  */
  enum class RtEntityId : int32_t {};

  /**
  Index type of entity ids.
  */
  using RtEntityIdIndex = std::underlying_type_t<RtEntityId>;

  /**
  Null entity id. It's undefined to write data to this id but valid to read. Any value
  read from this id is the default.
  */
  constexpr RtEntityId NullRtEntityId{ (RtEntityId)0 };

  /**
  The maximum entity id. This is the id a raytrace hit returns if an actual
  entity id hasn't been assigned the object being hit.
  */
  constexpr RtEntityId MaxRtEntityId{ (RtEntityId)( ( 1 << 28 ) - 1 ) };

  /**
  Index of the maximum enetity id.
  */
  constexpr RtEntityIdIndex MaxRtEntityIdIndex{ (RtEntityIdIndex)MaxRtEntityId };

  /**
  \param entityId - entity id to get index of
  \return the index of \p entityId
  */
  inline RtEntityIdIndex indexOf( RtEntityId entityId )
  {
    return (RtEntityIdIndex)entityId;
  }
}

namespace agx
{
  template<>
  struct HashFn<agxSensor::RtEntityId>
  {
    inline agxSensor::RtEntityIdIndex operator()( agxSensor::RtEntityId id ) const
    {
      return hash( agxSensor::indexOf( id ) );
    }
  };
}
