/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceOutputNoise.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES(RtDistanceGaussianNoise);

  /**
  This node applies a distance noise with gaussian distribution to the Lidar output data.
  Default perturbation is non-offset 0.02 m standard deviation growing at a rate of 0.0005 m/m, 
  seeded with 783u.
  Affects fields XYZ_VEC3_F32 and DISTANCE_F32.
  */
  class AGXSENSOR_EXPORT RtDistanceGaussianNoise : public RtOutputNoise
  {
    public:
      /**
      Default constructor using default gaussian distribution parameters for mean, stdDevBase and
      stdDevSlope.
      The standard deviation is calculated as s = stdDevBase + d * stdDevSlope where d is the
      distance in meters.
      */
      RtDistanceGaussianNoise() = default;

      /**
      Construct this distance noise given \p mean, \p stdDevBase and \p stdDevSlope.
      The standard deviation is calculated as s = stdDevBase + d * stdDevSlope where d is
      the distance in meters.
      \param mean - mean value for the gaussian noise
      \param stdDevBase - standard deviation base for the gaussian noise
      \param stdDevSlope - standard deviation slope, how the standard deviation changes with distance
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      RtDistanceGaussianNoise( agx::Real mean, agx::Real stdDevBase, agx::Real stdDevSlope,
                               agx::UInt32 seed = 783u );

      /**
      Set the mean value of the gaussian noise
      \param mean - mean value of the gaussian noise
      */
      void setMean( agx::Real mean );

      /**
      \return the mean value of the gaussian noise
      */
      agx::Real getMean() const;

      /**
      Set the standard deviation base value of the gaussian noise.
      \param stdDevBase - standard deviation base for the gaussian noise
      */
      void setStdDevBase( agx::Real stdDevBase );

      /**
      \return the standard deviation base value of the gaussian noise
      */
      agx::Real getStdDevBase() const;

      /**
      Set the standard deviation slope for the gaussian noise.
      \param stdDevSlope - the slope that controls how the standard deviation changes with distance
      */
      void setStdDevSlope( agx::Real stdDevSlope );

      /**
      \return the standard deviation slope of the gaussian noise, which determines how
      the standard deviation increases with distance
      */
      agx::Real getStdDevSlope() const;

      /**
      Set seed used to configure the pseudo-random perturbation generation.
      \param seed - pseudo-random perturbation generator seed
      */
      void setSeed( agx::UInt32 seed );

      /**
      \return the seed,  used to configure the pseudo-random perturbation generation
      */
      agx::UInt32 getSeed() const;

      /**
      Set all noise parameters at once: mean, standard deviation base, and standard deviation slope.
      \param mean - mean value of the gaussian noise
      \param stdDevBase - base standard deviation value of the gaussian noise
      \param stdDevSlope - slope that adjusts how the standard deviation changes with distance
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      void setNoise( agx::Real mean, agx::Real stdDevBase, agx::Real stdDevSlope, agx::UInt32 seed = 783u );

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~RtDistanceGaussianNoise() = default;
      virtual RtNode* createNode() override;
      virtual void update(RtNode* node, const CallbackData& data) override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtDistanceGaussianNoise );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      RtNode* update( RtNode* node );

    private:
      agx::Real m_mean{ 0.0 };
      agx::Real m_stdDevBase{ 0.02 };
      agx::Real m_stdDevSlope{ 0.0005 };
      agx::UInt32 m_seed{ 783u };
  };
}
