/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/Environment.h>
#include <agxSensor/ConstraintAttachedSensor.h>
#include <agxSensor/OdometerModel.h>
#include <agxSensor/OdometerOutputHandler.h>

#include <agx/CylindricalJoint.h>
#include <agx/Hinge.h>
#include <agx/SplineJoint.h>
#include <agx/SlackCylindricalJoint.h>
#include <agx/SlackHingeJoint.h>

#include <agxVehicle/WheelJoint.h>

#include <agxSDK/Simulation.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( Odometer );
  AGX_DECLARE_VECTOR_TYPES( Odometer );

  /**
  Odometer instance class defined by an attachment constraint and a model. Any output is accessed
  through the OdometerOutputHandler.
  \note This odometer represents a regular, wheel encoder-based odometer.
  */
  class AGXSENSOR_EXPORT Odometer : public ConstraintAttachedSensor
  {
    public:
      /**
      Find odometer given name.
      \param simulation - simulation with a sensor environment
      \param name - name of odometer instance to search for
      \return first odometer instance with name \p name if found, otherwise nullptr
      */
      static Odometer* find( const agxSDK::Simulation* simulation, const agx::Name& name );

      /**
      Find odometer given name.
      \param environment - sensor environment
      \param name - name of odometer instance to search for
      \return first odometer instance with name \p name if found, otherwise nullptr
      */
      static Odometer* find( const Environment* environment, const agx::Name& name );

      /**
      Find all odometers, given name.
      \param simulation - simulation with a sensor environment
      \param name - name of odometer instance to search for
      \return odometer instances with name \p name
      */
      static OdometerPtrVector findAll( const agxSDK::Simulation* simulation,
                                        const agx::Name& name );

      /**
      Find all odometers, given name.
      \param environment - sensor environment
      \param name - name of odometer instance to search for
      \return odometer instances with name \p name
      */
      static OdometerPtrVector findAll( const Environment* environment, const agx::Name& name );

      /**
      Find all odometers.
      \param simulation - simulation with a sensor environment
      \return all odometer instances in the sensor environment
      */
      static OdometerPtrVector findAll( const agxSDK::Simulation* simulation );

      /**
      Find all odometers.
      \param environment - sensor environment
      \return all odometer instances in the given sensor environment \p environment
      */
      static OdometerPtrVector findAll( const Environment* environment );

    public:
      /**
      Construct this odometer with the given \p model and attached to the given cylindrical joint
      constraint.
      \note This odometer instance is invalid if the given \p model is nullptr.
      \note This odometer instance only samples the rotational DoF of the cylindrical joint.
      \param cylindrical - cylindrical constraint to attach to and sample from
      \param model - model defining the parameters of this odometer instance, invalid if nullptr
      */
      Odometer( agx::CylindricalJoint* cylindrical, OdometerModel* model );

      /**
      Construct this odometer with the given \p model and attached to the given hinge constraint.
      \note This odometer instance is invalid if the given \p model is nullptr.
      \param hinge - hinge constraint to attach to and sample from
      \param model - model defining the parameters of this odometer instance, invalid if nullptr
      */
      Odometer( agx::Hinge* hinge, OdometerModel* model );

      /**
      Construct this odometer with the given \p model and attached to the given spline constraint.
      \note This odometer instance is invalid if the given \p model is nullptr.
      \note This odometer instance only samples the rotational DoF of the spline joint.
      \param spline - spline constraint to attach to and sample from
      \param model - model defining the parameters of this odometer instance, invalid if nullptr
      */
      Odometer( agx::SplineJoint* spline, OdometerModel* model );

      /**
      Construct this odometer with the given \p model and attached to the given slack cylindrical
      joint constraint.
      \note This odometer instance is invalid if the given \p model is nullptr.
      \note This odometer instance only samples the rotational DoF of the slack cylindrical joint.
      \param cylindrical - slack cylindrical constraint to attach to and sample from
      \param model - model defining the parameters of this odometer instance, invalid if nullptr
      */
      Odometer( agx::SlackCylindricalJoint* cylindrical, OdometerModel* model );

      /**
      Construct this odometer with the given \p model and attached to the given slack hinge
      constraint.
      \note This odometer instance is invalid if the given \p model is nullptr.
      \param hinge - slack hinge constraint to attach to and sample from
      \param model - model defining the parameters of this odometer instance, invalid if nullptr
      */
      Odometer( agx::SlackHingeJoint* hinge, OdometerModel* model );

      /**
      Construct this odometer with the given \p model and attached to the given wheel joint
      constraint.
      \note This odometer instance is invalid if the given \p model is nullptr.
      \note This odometer instance only samples the wheel rotation of the wheel joint.
      \param wheel - wheel constraint to attach to and sample from
      \param model - model defining the parameters of this odometer instance, invalid if nullptr
      */
      Odometer( agxVehicle::WheelJoint* wheel, OdometerModel* model );

      /**
      Assign a new parent cylindrical joint to attach this odometer to. The effects will take place
      the next time the environment is stepped.
      \note This odometer instance only samples the rotational DoF of the cylindrical joint.
      \param cylindrical - cylindrical constraint to attach to and sample from
      */
      void setConstraint( agx::CylindricalJoint* cylindrical );

      /**
      Assign a new parent hinge joint to attach this odometer to. The effects will take place
      the next time the environment is stepped.
      \param hinge - hinge constraint to attach to and sample from
      */
      void setConstraint( agx::Hinge* hinge );

      /**
      Assign a new parent spline joint to attach this odometer to. The effects will take place
      the next time the environment is stepped.
      \note This odometer instance only samples the rotational DoF of the spline joint.
      \param spline - spline constraint to attach to and sample from
      */
      void setConstraint( agx::SplineJoint* spline );

      /**
      Assign a new parent slack cylindrical joint to attach this odometer to. The effects will take
      place the next time the environment is stepped.
      \note This odometer instance only samples the rotational DoF of the slack cylindrical joint.
      \param cylindrical - slack cylindrical constraint to attach to and sample from
      */
      void setConstraint( agx::SlackCylindricalJoint* cylindrical );

      /**
      Assign a new parent slack hinge joint to attach this odometer to. The effects will take place
      the next time the environment is stepped.
      \param hinge - slack hinge constraint to attach to and sample from
      */
      void setConstraint( agx::SlackHingeJoint* hinge );

      /**
      Assign a new parent wheel joint to attach this odometer to. The effects will take place
      the next time the environment is stepped.
      \note This odometer instance only samples the wheel rotation of the wheel joint.
      \param wheel - wheel constraint to attach to and sample from
      */
      void setConstraint( agxVehicle::WheelJoint* wheel );

      /**
      \return the constraint of this odometer, defining what the odometer is attached to
      */
      virtual agx::Constraint* getConstraint() const override;

      /**
      \return the model of this odometer
      */
      OdometerModel* getModel() const;

      /**
      \return the output handler managing the outputs from the odometer
      */
      OdometerOutputHandler* getOutputHandler() const;

      /**
      \return the output handler managing the outputs from the odometer
      */
      virtual ISensorOutputHandler* getOutputHandlerBase() const override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void result( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::Odometer );

    protected:
      friend class OdometerOutputHandler;

      Odometer( agx::Constraint* constraint, OdometerModel* model, agx::UInt sampleDof );
      Odometer();

      virtual void setConstraint( agx::Constraint* constraint ) override;
      agx::UInt getConstraintSampleDof() const;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::ConstraintRef m_constraint;
      agx::UInt m_constraintSampleDof;
      OdometerModelRef m_model;
  };
}
