/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/TriaxialGaussianNoise.h>
#include <agxSensor/TriaxialOutputHandler.h>
#include <agxSensor/TriaxialSignalScaling.h>
#include <agxSensor/TriaxialSpectralGaussianNoise.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( MagnetometerOutputHandler );

  /**
  Handler responsible for how the magnetometer readings are assembled and provided to the user.
  */
  class AGXSENSOR_EXPORT MagnetometerOutputHandler : public TriaxialOutputHandler
  {
    public:
      /**
      Default constructor.
      */
      MagnetometerOutputHandler();

      /**
      Add a Gaussian noise instance to the system tree.
      \param noise - output noise to add to the system tree
      \return true if added successfully, false if nullptr or already added
      */
      bool add( TriaxialGaussianNoise* noise );

      /**
      Remove the Gaussian noise instance from the system tree. The noise will no longer be applied
      to the output data if removed successfully.
      \param noise - noise to remove
      \return true if removed successfully, false if nullptr or not part of this
              system tree
      */
      bool remove( TriaxialGaussianNoise* noise );

      /**
      \param noise - noise instance to check if present and active
      \return true if \p noise is part of this output handler
      */
      bool contains( const TriaxialGaussianNoise* noise ) const;

      /**
      Add a signal scaling instance to the system tree.
      \param scaling - signal scaling to add to the system tree
      \return true if added successfully, false if nullptr or already added
      */
      bool add( TriaxialSignalScaling* scaling );

      /**
      Remove the signal scaling instance from the system tree. The scaling will no longer be applied
      to the output data if removed successfully.
      \param scaling - scaling to remove
      \return true if removed successfully, false if nullptr or not part of this
              system tree
      */
      bool remove( TriaxialSignalScaling* scaling );

      /**
      \param scaling - signal scaling instance to check if present and active
      \return true if \p scaling is part of this output handler
      */
      bool contains( const TriaxialSignalScaling* scaling ) const;

      /**
      Add a noise density based noise instance to the system tree.
      \param noise - output noise to add to the system tree
      \return true if added successfully, false if nullptr or already added
      */
      bool add( TriaxialSpectralGaussianNoise* noise );

      /**
      Remove the noise density based  noise instance from the system tree. The noise will no longer
      be applied to the output data if removed successfully.
      \param noise - noise to remove
      \return true if removed successfully, false if nullptr or not part of this
              system tree
      */
      bool remove( TriaxialSpectralGaussianNoise* noise );

      /**
      \param noise - noise instance to check if present and active
      \return true if \p noise is part of this output handler
      */
      bool contains( const TriaxialSpectralGaussianNoise* noise ) const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      using TriaxialOutputHandler::add;

    public:
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::MagnetometerOutputHandler );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      SystemNodeRef m_signalAssembler;
  };
}
