/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxSensor/LidarRayDistortion.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES(LidarRayAngleGaussianNoise);

  /**
  Gaussian distributed perturbation of lidar ray emission and detection angle.
  Default perturbation is non-offset 0.01 radians standard deviation around Y (horizontal
  perturbation) seeded with 738u.
  */
  class AGXSENSOR_EXPORT LidarRayAngleGaussianNoise : public LidarRayDistortion
  {
    public:
      /**
      Axis, local to the ray frame, around which the perturbation should be applied.
      */
      enum Axis : agx::UInt8
      {
        AXIS_X = 1,
        AXIS_Y = 2,
        AXIS_Z = 3
      };

    public:
      /**
      Default constructor, where the default perturbation is non-offset 0.01 radians standard
      deviation around Y (horizontal perturbation).
      */
      LidarRayAngleGaussianNoise() = default;

      /**
      Construct using the given perturbation parameters.
      \param mean - mean of the Gaussian random perturbation in radians
      \param std - standard deviation of the Gaussian random perturbation in radians
      \param axis - axis, local to the ray frame, around which the perturbation should be applied
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      explicit LidarRayAngleGaussianNoise( agx::Real mean, agx::Real std,
                                           LidarRayAngleGaussianNoise::Axis axis,
                                           agx::UInt32 seed = 783u );

      /**
      Construct using the given perturbation parameters.
      \param mean - mean of the Gaussian random perturbation in radians
      \param std - standard deviation of the Gaussian random perturbation in radians
      \param axisVector - axis, local to the ray frame, around which the perturbation should be
                          applied
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      explicit LidarRayAngleGaussianNoise( agx::Real mean, agx::Real std, agx::Vec3 axisVector,
                                           agx::UInt32 seed = 783u );

      /**
      Set new mean of the Gaussian random perturbation applied to the rays.
      \param mean - new Gaussian random perturbation mean in radians
      */
      void setMean( agx::Real mean );

      /**
      \return the mean of the Gaussian random perturbation applied to the rays
      */
      agx::Real getMean() const;

      /**
      Set new standard deviation of the Gaussian random perturbation applied to the rays.
      \param std - new Gaussian random perturbation standard deviation in radians
      */
      void setStandardDeviation( agx::Real std );

      /**
      \return the standard deviation of the Gaussian random perturbation applied to the rays
      */
      agx::Real getStandardDeviation() const;

      /**
      Set axis, local to the ray frame, around which the perturbation should be applied.
      \param axis - new axis around which the perturbation should be applied
      */
      void setAxis( LidarRayAngleGaussianNoise::Axis axis );

      /**
      \return the closest basis axis, local to the ray frame, around which the perturbation is applied
      */
      LidarRayAngleGaussianNoise::Axis getAxis() const;

      /**
      Set axis, local to the ray frame, around which the perturbation should be applied.
      \param axisVector - new axis around which the perturbation should be applied
      */
      void setAxisVector( const agx::Vec3& axisVector );

      /**
      \return the axis, local to the ray frame, around which the perturbation is applied
      */
      const agx::Vec3& getAxisVector() const;

      /**
      Set seed used to configure the pseudo-random perturbation generation.
      \param seed - pseudo-random perturbation generator seed
      */
      void setSeed( agx::UInt32 seed );

      /**
      \return the seed,  used to configure the pseudo-random perturbation generation
      */
      agx::UInt32 getSeed() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual RtNode* createNode() override;
      virtual void synchronize(RtNode* node, const CallbackData& data) override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarRayAngleGaussianNoise );

    private:
      RtNode* updatePerturbation(RtNode* node);

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::Real m_mean = 0.0f;
      agx::Real m_std = 0.01f;
      agx::Vec3 m_axisVector = { 1.0, 0.0, 0.0 };
      agx::UInt32 m_seed = 783u;
  };
}

