/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/AccelerometerModel.h>
#include <agxSensor/AccelerometerOutputHandler.h>
#include <agxSensor/Environment.h>
#include <agxSensor/FrameAttachedSensor.h>

#include <agx/Frame.h>

#include <agxSDK/Simulation.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( Accelerometer );
  AGX_DECLARE_VECTOR_TYPES( Accelerometer );

  /**
  Accelerometer instance class defined by a frame/transform and a model. Any output from the sensor
  is accessed through the AccelerometerOutputHandler.
  */
  class AGXSENSOR_EXPORT Accelerometer : public FrameAttachedSensor
  {
    public:
      /**
      Find accelerometer given name.
      \param simulation - simulation with a sensor environment
      \param name - name of accelerometer instance to search for
      \return first accelerometer instance with name \p name if found, otherwise nullptr
      */
      static Accelerometer* find( const agxSDK::Simulation* simulation, const agx::Name& name );

      /**
      Find accelerometer given name.
      \param environment - sensor environment
      \param name - name of accelerometer instance to search for
      \return first accelerometer instance with name \p name if found, otherwise nullptr
      */
      static Accelerometer* find( const Environment* environment, const agx::Name& name );

      /**
      Find all accelerometers, given name.
      \param simulation - simulation with a sensor environment
      \param name - name of accelerometer instance to search for
      \return accelerometer instances with name \p name
      */
      static AccelerometerPtrVector findAll( const agxSDK::Simulation* simulation,
                                             const agx::Name& name );

      /**
      Find all accelerometers, given name.
      \param environment - sensor environment
      \param name - name of accelerometer instance to search for
      \return accelerometer instances with name \p name
      */
      static AccelerometerPtrVector findAll( const Environment* environment, const agx::Name& name );

      /**
      Find all accelerometers.
      \param simulation - simulation with a sensor environment
      \return all accelerometer instances in the sensor environment
      */
      static AccelerometerPtrVector findAll( const agxSDK::Simulation* simulation );

      /**
      Find all accelerometers.
      \param environment - sensor environment
      \return all accelerometer instances in the given sensor environment \p environment
      */
      static AccelerometerPtrVector findAll( const Environment* environment );

    public:
      /**
      Construct this accelerometer attached to the given \p frame and configured by the \p model.
      \note This accelerometer instance is invalid if the given \p model is nullptr.
      \param frame - parent frame to attach this accelerometer to, may be dynamic, the output is
                     relative to this frame. If nullptr, a new frame is created aligned to the world
                     frame
      \param model - model defining the parameters of this accelerometer instance, invalid if
                     nullptr
      */
      Accelerometer( agx::Frame* frame, AccelerometerModel* model );

      /**
      Assign a new parent frame to attach this accelerometer to. The effects will take place the
      next time the environment is stepped.
      \param frame - new frame for this accelerometer
      */
      virtual void setFrame( agx::Frame* frame ) override;

      /**
      \return the frame of this accelerometer, defining this accelerometer -> world transform
      */
      virtual agx::Frame* getFrame() const override;

      /**
      \return the model of this accelerometer
      */
      AccelerometerModel* getModel() const;

      /**
      \return the output handler managing the outputs from the accelerometer
      */
      AccelerometerOutputHandler* getOutputHandler() const;

      /**
      \return the output handler managing the outputs from the accelerometer
      */
      virtual ISensorOutputHandler* getOutputHandlerBase() const override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void result( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::Accelerometer );

    protected:
      Accelerometer();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::FrameRef m_frame;
      AccelerometerModelRef m_model;
  };
}
