/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or
having been advised so by Algoryx Simulation AB for a time limited evaluation,
or having purchased a valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Referenced.h>
#include <agx/HashTable.h>
#include <agxPowerLine/PowerLine.h>

namespace agxPowerLine
{
  class PowerLine;

  AGX_DECLARE_POINTER_TYPES( DotGraphWriter );


  class AGXMODEL_EXPORT DotGraphWriter : public agx::Referenced
  {
  public:
    DotGraphWriter(const char* rotRateQuantity = "rad/s", const char* transRateQuantity = "m/s");

    void registerQuantity(const char* dimensionName, const char* quantityName);

    bool writeDimensionsGraph(const agx::String& filename, const PowerLine* powerline);
    bool writeDimensionsGraph(const agx::String& filename, const agxPowerLine::Unit* startUnit);
    bool writeDimensionsGraph(const agx::String& filename, const UnitPtrPowerLinePtrHashVector& roots);

    //bool writeUnitGraph(const char* filename, agxPowerLine::PowerLine* powerline);

    // Each Physical Dimension and Connector belong to a Sub Graph. This struct
    // keeps track of those relationships. We only track top-level Sub Graphs,
    // so any nesting is flattened when this is being built.
    struct SeenSubGraph {
      const SubGraph* subGraph;
      // These contain indices pointing into m_seenDimensions and m_seenConnectors.
      agx::HashSet<size_t> dimensions;
      agx::HashSet<size_t> connectors;

      SeenSubGraph(const SubGraph* inSubGraph)
        : subGraph {inSubGraph}
      {
        if (subGraph->is<Unit>())
        {
          // Most Units have only 1 Physical Dimension.
          dimensions.reserve(1);
        }
      }
    };

  protected:
    virtual ~DotGraphWriter() {}

  private:
    void clear();
    bool writeDimensionsGraph(const agx::String& filename);
    void visitDimensionConnections( const agxPowerLine::ConnectionRefVector& connections );
    void visitConnectorDimensions( const agxPowerLine::ConnectionRefVector& connections );
    struct SeenSubGraph& getSeenRootSubGraph(const SubGraph* subGraph);

  private:
    typedef agx::HashTable<agx::String, agx::String> QuantitiesTable;
    QuantitiesTable m_quantities;


    typedef agx::Vector<const PhysicalDimension*> DimensionVector;
    typedef agx::Vector<const Connector*> ConnectorVector;

    typedef agx::Vector<SeenSubGraph> SeenSubGraphVector;
    SeenSubGraphVector m_seenSubGraphs;

    // The index of each dimension and connector in these vectors determines it's
    // ID within the DOT graph. Each dimension and connector may only be represented
    // once.
    DimensionVector m_seenDimensions;
    ConnectorVector m_seenConnectors;

    typedef agx::HashTable<const SubGraph*, size_t> SubGraphToIndexTable;
    typedef agx::HashTable<const PhysicalDimension*, size_t> DimensionToIndexTable;
    typedef agx::HashTable<const Connector*, size_t> ConnectorToIndexTable;

    // The index of each unit, dimension and connector is kept in these hash tables.
    SubGraphToIndexTable m_subGraphToIndexTable;
    DimensionToIndexTable m_dimensionToIndexTable;
    ConnectorToIndexTable m_connectorToIndexTable;

    typedef agx::Vector<const Unit*> UnitVector;
    UnitVector m_queue;
  };
}
