/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/IGenericConstraint.h>
#include <agx/GenericConstraintData.h>

namespace agx
{
  /**
  Generic constraint which by default is empty of elementary constraints.
  This enables the possibility to build custom/generic constraints with
  additional GenericElementaryConstraint instances or to use the already
  implemented in the toolkit.

  This constraint receives; onAddNotification, onRemoveNotification, onPreSolve and onPostSolve.

  Python example:
    class MyHinge(agx.GenericConstraint):
        def __init__(self, rb1: agx.RigidBody, rb1Frame: agx.Frame, rb2: agx.RigidBody, rb2Frame: agx.Frame):
            super().__init__(rb1, rb1Frame, rb2, rb2Frame)

            self.addElementaryConstraint("SP", agx.SphericalRel(agx.ElementaryConstraintData(self.getAttachmentPair())))

            # See include/agx/GenericElementaryConstraint.h for an implementation of MyDot1.
            self.addElementaryConstraint("D1VN", MyDot1(agx.Attachment.V, agx.Attachment.N))
            self.addElementaryConstraint("D1UN", MyDot1(agx.Attachment.U, agx.Attachment.N))

        def onAddNotification(self):
            print('add notification!')

        def onRemoveNotification(self):
            print('remove notification!')

        def onPreSolve(self):
            # Return True to signal this constraint is active and should be passed to the solver.
            return True

        def onPostSolve(self):
            # Post-processing or something.
            pass
  */
  class AGXPHYSICS_EXPORT GenericConstraint : public IGenericConstraint
  {
    public:
      /**
      Construct given one rigid body (second is null).
      \note Two frames will be created and added to the AttachmentPair of this constraint.
            The transform of the constraint frames is identity.
      \param rb1 - first rigid body
      */
      GenericConstraint( agx::RigidBody* rb1 );

      /**
      Construct given two rigid bodies (first has to be != null).
      \note Two frames will be created and added to the AttachmentPair of this constraint.
            The transform of the constraint frames is identity.
      \param rb1 - first rigid body
      \param rb2 - second rigid body
      */
      GenericConstraint( agx::RigidBody* rb1, agx::RigidBody* rb2 );

      /**
      Construct given two bodies and frames.
      \param rb1 - first rigid body
      \param f1 - constraint frame relative to rb1
      \param rb2 - second rigid body
      \param f2 - constraint frame relative to rb2
      */
      GenericConstraint( agx::RigidBody* rb1, agx::Frame* f1, agx::RigidBody* rb2, agx::Frame* f2 );

    public:
      /**
      Callback when the constraint has been added to a simulation.
      */
      virtual void onAddNotification() override;

      /**
      Callback when the constraint has been removed from a simulation.
      */
      virtual void onRemoveNotification() override;

      /**
      Callback before this constraint reaches the solver (actually preSystemCallback). If the
      return value is true and this constraint is valid, this constraint will continue into
      the solver.
      \return true if this constraint is active, false if inactive
      */
      virtual agx::Bool onPreSolve() override;

      /**
      Callback from after the solve and the solutions been pushed to the elementary constraints.
      */
      virtual void onPostSolve() override;

      /**
      Callback to write data. All values must be greater than or equal to zero. Data is
      only stored for the lower triangle since the matrix must be symmetric and positive
      definite.
      */
      virtual void updateComplianceMatrix( GenericConstraintComplianceMatrix* matrix ) override;

    public:
      int getNumDOF() const override;
      void render( class agxRender::RenderManager* manager, float scale ) const override;

    protected:
      virtual ~GenericConstraint() = default;
  };
}
