/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agx/Referenced.h>
#include <agx/Bound.h>
#include <agxCollide/Box.h>
#include <agxSDK/Simulation.h>

namespace agx
{
  AGX_DECLARE_POINTER_TYPES(AnalysisBox);

  /**
  Class that can be used to extract data from elements in a simulation within a specified bound.
  */
  class AGXPHYSICS_EXPORT AnalysisBox : public agx::Referenced
  {
  public:
    typedef HashSet<agx::Index> ParticleIdSet;

  public:
    /**
    Constructor for the AnalysisBox.

    \param simulation The simulation to couple the analysis box too.
    \param halfVec Half vector that spans the box.
    \param transform The transformation of the analysis box, translation and rotation.
    */
    AnalysisBox(agxSDK::Simulation* simulation,
                const agx::Vec3& halfVec,
                const agx::AffineMatrix4x4& transform=agx::AffineMatrix4x4());

    /// Update the analysis box contents.
    void updateContents(bool includeRigidBodies=false,
                        bool onlyIncludeEmittedBodies=false);

    /// Return the particle mass in the particles inside the Analysis Box.
    Real calculateParticleMassInside();

    /// Return the body mass in the particles inside the Analysis Box.
    Real calculateBodyMassInside( bool onlyIncludeEmittedBodies=false );

    /// Estimate the particle mass flow inside the Analysis Box.
    Real estimateMassFlowInside( bool includeParticles=true, bool includeBodies=true );

    /// Returns the particles ids inside the Analysis Box.
    ParticleIdSet getParticlesInside() const;

    /// Returns the rigid bodies inside the Analysis Box.
    RigidBodyPtrVector getBodiesInside() const;

    /// Returns true/false if the particle given the specified particle id is inside the box.
    bool containsParticle(agx::Index particleId) const;

    /// Returns true/false if the point is contained inside the Analysis Box bound.
    bool containsPoint(const agx::Vec3& point) const;

    /// Get number of particles in bound from last content update
    size_t getNumParticlesInside();

    /// Get particle mass inside bound from last content update
    agx::Real getParticleMassInside() const;

    /// Get number of rigid bodies in bound from last content update
    size_t getNumBodiesInside();

    /// Get rigid body mass inside bound from last content update
    agx::Real getBodyMassInside() const;

    /// Get total mass inside bound from last content update
    agx::Real getTotalMassInside() const;

    /// Return the estimated bulk density inside the Analysis Box based on the current contents.
    Real calculateBulkDensity() const;

    /// Return the volume of the Analysis Box.
    Real calculateBoxVolume() const;

    /// Return true/false if the box is enabled.
    bool getEnable() const;

    /// Set true/false to enable/enable the Analysis Box.
    void setEnable(bool enable);

    /// Get the center point of the Analysis Box bound.
    agx::Vec3 getBoxCenter() const;

    /// Get the half vector of the Analysis Box bound.
    agx::Vec3 getBoxHalfVec() const;

    /// Set the half vector of the Analysis Box bound.
    void setBoxHalfVec( const agx::Vec3& halfVec );

    /// Set the center of the Analysis Box bound.
    void setBoxCenter( const agx::Vec3& center );

    /// Set the rotation of the Analysis Box bound.
    void setBoxRotation( const agx::Quat& rotation );

    /// Get the rotation of the Analysis Box bound.
    agx::Quat getBoxRotation() const;

    /// Get the local position of the specified world position
    agx::Vec3 getLocalPositionInBox(const agx::Vec3& worldPosition) const;

    /// Get the transform of the Analysis box.
    agx::AffineMatrix4x4 getTransform() const;

    /// Return true/false if the box rendering is enabled.
    bool getEnableRendering() const;

    /// Set true/false to enable/enable rendering of the Analysis Box.
    void setEnableRendering( bool enable );

    /// Clears all content data inside the box
    void clearContentData();

  public:

    DOXYGEN_START_INTERNAL_BLOCK()

    void printContents();

    void setEnableDebugRenderContents( bool enable );

    bool getEnableDebugRenderContents() const;

    DOXYGEN_END_INTERNAL_BLOCK()

  protected:
    /// Virtual Destructor
    virtual ~AnalysisBox();

    void updateInvTransform();

    agx::AffineMatrix4x4 getInvTransform() const;

    agx::ParticleSystem * getParticleSystem();

    //////////////////////////////////////////////////////////////////////////
    // Variables
    //////////////////////////////////////////////////////////////////////////
  protected:
    agxSDK::Simulation*        m_simulation;
    agx::Bound3                m_bound;
    agx::AffineMatrix4x4       m_transform;
    agx::AffineMatrix4x4       m_invBoxTransform;
    RigidBodyPtrVector         m_rigidBodiesInside;
    ParticleIdSet              m_particleIdsInside;
    agx::Real                  m_particleMassInside;
    agx::Real                  m_bodyMassInside;
    bool                       m_enabled;
    bool                       m_enableRendering;
    bool                       m_enableDebugRenderContents;
  };

  AGX_FORCE_INLINE agx::AnalysisBox::ParticleIdSet AnalysisBox::getParticlesInside() const { return m_particleIdsInside; }

  AGX_FORCE_INLINE agx::RigidBodyPtrVector AnalysisBox::getBodiesInside() const { return m_rigidBodiesInside; }

  AGX_FORCE_INLINE bool AnalysisBox::getEnable() const { return m_enabled; }

  AGX_FORCE_INLINE void AnalysisBox::setEnable(bool enable) { m_enabled = enable; }

  AGX_FORCE_INLINE bool AnalysisBox::getEnableDebugRenderContents() const { return m_enableDebugRenderContents; }

  AGX_FORCE_INLINE void AnalysisBox::setEnableDebugRenderContents( bool enable ) { m_enableDebugRenderContents = enable; }

  AGX_FORCE_INLINE agx::AffineMatrix4x4 AnalysisBox::getTransform() const { return m_transform; }

  AGX_FORCE_INLINE agx::AffineMatrix4x4 AnalysisBox::getInvTransform() const { return m_invBoxTransform; }
}
