#pragma once

/*
 * Definitions for types used in both the 'api' part and the 'impl' part of
 * Algoryx GPU Sensors.
 *
 * Requires that some RGL types are available, so must be included after either
 * 'rgllike_types.h' (For the 'api' part.) or RGLs 'core.h' (For the 'impl'
 * part.).
 */


#if !defined(RGL_DEFAULT_ENTITY_ID)
	#error "Must include agpu/api/rgllike_types.h or rgl/api/core.h before agpu/common/types.h."
#endif

#include <cstdint>

extern "C" {

/**
 * Status / error codes produced by the API side, as opposed to the
 * implementation side of Algoryx GPU Sensors. The implementation side uses
 * rgl_status_t instead.
 */
enum agpu_status_t : int32_t
{
	AGPU_SUCCESS = 0,
	AGPU_CUDA_NOT_SUPPORTED,
	AGPU_LOAD_LIBRARY_ERROR
};



/**
 * The different sources of status / error codes Algoryx GPU Sensors can return.
 */
enum agpu_status_source_t : int32_t
{
	AGPU_STATUS_SOURCE_NONE,
	AGPU_STATUS_SOURCE_AGPU,
	AGPU_STATUS_SOURCE_RGL,
	AGPU_STATUS_SOURCE_NUM
};



/**
 * A container for a status / error code. Holds either a Robotec GPU Lidar
 * status or an Algoryx GPU Sensor status, along with a 'source' member
 * identifying which tpe of status is currently held.
 */
struct agpu_status_code_t
{
	agpu_status_source_t source;

	union {
		agpu_status_t agpu_status;
		rgl_status_t rgl_status;
	};
};

inline agpu_status_code_t agpu_make_agpu_status_code(agpu_status_t status)
{
	agpu_status_code_t code;
	code.source = AGPU_STATUS_SOURCE_AGPU;
	code.agpu_status = status;
	return code;
}

inline agpu_status_code_t agpu_make_rgl_status_code(rgl_status_t status)
{
	agpu_status_code_t code;
	code.source = AGPU_STATUS_SOURCE_RGL;
	code.rgl_status = status;
	return code;
}

inline bool agpu_status_code_is_success(agpu_status_code_t code)
{
	switch (code.source)
	{
		case AGPU_STATUS_SOURCE_AGPU:
			return code.agpu_status == AGPU_SUCCESS;
		case AGPU_STATUS_SOURCE_RGL:
			return code.rgl_status == RGL_SUCCESS;
		case AGPU_STATUS_SOURCE_NONE:
		case AGPU_STATUS_SOURCE_NUM:
			return false;
	}
	return false;
}

inline bool agpu_status_code_is_agpu(agpu_status_code_t code, agpu_status_t status)
{
	return code.source == AGPU_STATUS_SOURCE_AGPU && code.agpu_status == status;
}

inline bool agpu_status_code_is_rgl(agpu_status_code_t code, rgl_status_t status)
{
	return code.source == AGPU_STATUS_SOURCE_RGL && code.rgl_status == status;
}

}
