
#pragma once

#include <memory>
#include <unordered_set>
#include <unordered_map>
#include <openplx/PathNode.h>
#include <openplx/VarAssignment.h>
#include <openplx/Node.h>
#include <openplx/Type.h>

namespace openplx {
    class TopologicalPath : public std::enable_shared_from_this<TopologicalPath> {
        public:
            static std::shared_ptr<TopologicalPath> fromEmpty();
            static std::shared_ptr<TopologicalPath> fromNode(NodePtr node);
            static std::shared_ptr<TopologicalPath> fromNodes(const std::vector<NodePtr>& nodes);
            static std::shared_ptr<TopologicalPath> fromAppend(
                    const std::shared_ptr<TopologicalPath>& prefix,
                    NodePtr node);
            static std::shared_ptr<TopologicalPath> copy(const TopologicalPath& other, size_t offset = 0);
            static std::shared_ptr<TopologicalPath> copyUntil(const TopologicalPath& other, size_t size);

            static bool compare(TopologicalPath& lhs, TopologicalPath& rhs);
            static bool compareSymbols(TopologicalPath& lhs, TopologicalPath& rhs);
            static std::shared_ptr<TopologicalPath> commonPrefix(std::unordered_set<std::shared_ptr<TopologicalPath>>& paths);

            bool compareSymbols(TopologicalPath& other);
            const std::vector<PathNode>& getPath() const;
            std::shared_ptr<TopologicalPath> calculateThisPath() const;
            size_t symbolCount() const;
            std::string symbolAt(size_t index) const;
            bool detectLoop() const;

            void appendNode(NodePtr node);
            void appendLastSymbol(VarAssignPtr var_assignment);
            void pop();
            void popOneSymbol();

            std::string toString() const;
            std::string toSymbolString() const;
            size_t getSymbolCount() const;

            ExprPtr getValue() const;
            TypePtr getType() const;

            std::string getRoot() const;

            bool isDefault() const;
            bool isEmpty() const;

            void setOuterTypes(const std::vector<TypePtr>& outer_types);

            void unbind();

            /* Degree */
            int64_t getDegree() const;
            void updateDegree(int64_t degree);

            /* Valid */
            bool isValid() const;
            void setValid(bool valid);

            /* Edges */
            void addEdge(std::shared_ptr<TopologicalPath> edge);
            const std::unordered_set<std::shared_ptr<TopologicalPath>>& getEdges() const;

        private:
            TopologicalPath();

            std::vector<PathNode> m_path;
            size_t m_symbol_count;
            std::unordered_set<std::shared_ptr<TopologicalPath>> m_edges;
            int64_t m_degree;
            bool m_valid;
    };

    using TopoPathPtr = std::shared_ptr<TopologicalPath>;
}
