
#pragma once

#include <memory>
#include <string>
#include <vector>
#include <map>
#include <unordered_map>
#include <unordered_set>
#include <openplx/Token.h>
#include <openplx/Node.h>
#include <openplx/CompositeType.h>
#include <openplx/TopologicalPath.h>
#include <openplx/TreeNode.h>

namespace openplx
{
    class ModelDeclaration : public CompositeType, public std::enable_shared_from_this<ModelDeclaration>
    {
        public:
            /* Factories */
            static ModelDeclPtr create(const Token& qualifier_token,
                                       const Token& name_token,
                                       const Token& extends_qualifier_token,
                                       const std::vector<Token>& parent_segments,
                                       const std::vector<AnnotationPtr>& annotations,
                                       const std::vector<TraitImplPtr>& traits,
                                       const std::vector<DeletionPtr>& deletions,
                                       const std::vector<NodePtr>& members);

            /* Overrides */
            ModelDeclPtr asModelDeclaration() override;
            TypePtr asType() override;
            CompositeTypePtr asCompositeType() override;
            void accept(NodeVisitor& visitor) override;
            bool isAssignableTo(const TypePtr &other) override;
            std::string toString() override;
            std::string toKey() override;
            void unbind() override;

            bool isEmpty() const override;
            VarAssignPtr findFirstMemberWithType(const std::string& name) const override;
            VarAssignPtr lookupDeclaration(const std::string& symbol_path) override;
            VarAssignPtr lookupDeclarationFromSegments(const std::vector<Token>& segments, size_t first_ix, size_t last_ix) override;

            /* Qualifier */
            Token getQualifierToken() const;
            bool isConst() const;
            bool isTrait() const;

            /* Name */
            const Token& getNameToken() const;
            const std::string& getName() const;
            std::string getNameWithNamespace(const std::string& separator) const;
            std::string getNameWithNamespaceWithPrefix(const std::string& separator, const std::string& prefix = "openplx") const;
            std::string getNameWithNamespaceSkipFirst(const std::string& separator) const;

            /* Parent qualifier token */
            const Token& getExtendsQualifierToken() const;
            bool isExternal() const;

            /* Parent */
            std::string extendsSegmentsAsString() const;
            const std::vector<Token>& getExtendsSegments() const;
            void replaceExternalExtendsSegments(const std::vector<Token>& new_segments);
            ModelDeclPtr getExtends() const;
            void setExtends(ModelDeclPtr extends);

            /* Annotations */
            void appendToAnnotations(AnnotationPtr annotation);
            const std::vector<AnnotationPtr>& getAnnotations() const;
            std::vector<AnnotationPtr> findModelAnnotations(const std::string& name) const;

            /* Traits */
            void appendToTraits(TraitImplPtr trait);
            const std::vector<TraitImplPtr>& getTraits() const;

            /* Deletes */
            void appendToDeletions(DeletionPtr trait);
            const std::vector<DeletionPtr>& getDeletions() const;

            /* Members */
            void appendToMembers(NodePtr member);
            void extendMembers(const std::vector<NodePtr>& members);
            const std::vector<NodePtr>& getMembers() const;
            std::vector<NodePtr> getOuterMembers() const;
            std::vector<NodePtr> collapseMembers() const;
            void removeMember(const NodePtr& member);
            void removeInvalidMembers();
            std::vector<NodePtr> findMembers(const std::string& name) const;
            std::vector<NodePtr> findMembersOfType(NodeType type) const;
            std::vector<VarAssignPtr> findAttributesInOrder() const;
            NodePtr findFirstMemberOfType(const std::string& name, NodeType type) const;
            NodePtr findFirstMemberExcludeType(const std::string& name, NodeType type) const;
            NodePtr findFirstMember(const std::string& name) const;
            size_t countMembers() const;
            bool isSyntacticallyEmpty() const;
            bool isInitializable() const;

            /* Symbol map */
            void setIsNewSymbol(const std::string& symbol_path, NodePtr symbol);
            bool hasNewSymbol(const std::string& symbol_path);
            void setIsExtendedSymbol(const std::string& symbol_path);
            bool isExtendedSymbol(const std::string& symbol_path) const;
            void setIsReferenceSymbol(const std::string& symbol_path);
            bool isReferenceSymbol(const std::string& symbol_path) const;

            /* Owning document */
            DocPtr getOwningDocument() const;
            void setOwningDocument(DocPtr owning_document);
            std::string getSourceIdOrDefault() const;

            /* Symbol tree */
            std::shared_ptr<TreeNode<VarAssignment>> getInnerTreeRoot() const;
            std::shared_ptr<TreeNode<SymbolMetaData>> getOuterTreeRoot() const;

            /* Topological sort */
            const std::vector<TopoPathPtr> getTopologicalSort() const;
            void setTopologicalSort(const std::vector<TopoPathPtr>& topological_sort);

        private:
            void collectOuterMembers(std::map<std::string, NodePtr>& members) const;
            void collectMembersOfType(NodeType type, std::vector<NodePtr>& output) const;
            void collectAttributesInOrder(std::vector<VarAssignPtr>& output, std::unordered_map<std::string, VarAssignPtr>& outer_declarations) const;
            void collectCollapsedMembers(std::vector<NodePtr>& output) const;
            ModelDeclaration();
            Token m_qualifier_token;
            Token m_name_token;
            Token m_extends_qualifier_token;
            std::vector<Token> m_extends_segments;
            ModelDeclPtr m_extends;
            std::vector<AnnotationPtr> m_annotations;
            std::vector<TraitImplPtr> m_traits;
            std::vector<DeletionPtr> m_deletions;
            std::vector<NodePtr> m_members;
            DocPtr m_owning_document;
            std::vector<TopoPathPtr> m_topological_sort;
            std::vector<std::string> m_unbound_namespace;
            std::unordered_map<std::string, NodePtr> m_new_symbols;
            std::unordered_set<std::string> m_extended_symbols;
            std::unordered_set<std::string> m_reference_symbols;
            std::shared_ptr<TreeNode<VarAssignment>> m_inner_tree_root;
            std::shared_ptr<TreeNode<SymbolMetaData>> m_outer_tree_root;
    };
}

