
#pragma once

#include <memory>
#include <stack>
#include <unordered_map>
#include <openplx/Any.h>
#include <openplx/EvaluatorContext.h>
#include <openplx/Object.h>
#include <openplx/NodeVisitor.h>
#include <openplx/Array.h>
#include <openplx/BinaryOp.h>
#include <openplx/Call.h>
#include <openplx/Constant.h>
#include <openplx/Document.h>
#include <openplx/Expression.h>
#include <openplx/MemberAccess.h>
#include <openplx/MethodDeclaration.h>
#include <openplx/ModelDeclaration.h>
#include <openplx/Node.h>
#include <openplx/Parameter.h>
#include <openplx/PrimitiveType.h>
#include <openplx/Unary.h>
#include <openplx/Type.h>
#include <openplx/VarAssignment.h>
#include <openplx/ErrorReporter.h>

namespace openplx::Core {

    /**
     * Internal visitor for evaluating modeltree expressions
     */
    class ExpressionEvaluator : private NodeVisitor
    {
        public:
            ExpressionEvaluator(
                std::shared_ptr<EvaluatorContext> ctx,
                std::shared_ptr<ErrorReporter> error_reporter);

            Any evalDynamic(const ExprPtr& expression, const TypePtr& expected_type);

            void setThis(const std::shared_ptr<Object>& curr_this) {
                m_this = curr_this;
            }

            void setCurrentDocument(const DocPtr& document) {
                this->m_current_document = document;
            }
            std::shared_ptr<Core::Object> createInstance(const CompositeTypePtr& composite_type, ObjectPtr owner);

            const std::unordered_map<ModelDeclPtr, std::shared_ptr<Object>>& getSingletons() const;
        private:
            void visitConstant(const ConstantPtr &constant) override;

            void visitCall(const CallPtr &call) override;

            void visitMemberAccess(const MemberAccessPtr &member_access) override;

            void visitArray(const ArrayPtr &array) override;

            void visitInitializer(const InitializerPtr& initializer) override;

            void visitUnary(const UnaryPtr &unary) override;

            void visitBinaryOp(const BinaryOpPtr &binary_op) override;

            void visitIndexing(const IndexingPtr& indexing) override;

            std::optional<Core::FactoryMethod> lookupExternalFactoryMethod(const ModelDeclPtr& model_declaration);

            void addReflectionData(const Any& target,
                                   const TypePtr& type);

            // Expression stack, the expression that is currently being evaluated
            std::stack<Any> m_stack;
            // Stack of current evaluated model, in order to resolve "this" and implicit "this"
            std::shared_ptr<Object> m_this;
            std::shared_ptr<EvaluatorContext> m_context;
            std::stack<TypePtr> m_type_stack;

            // Singletons
            std::unordered_map<ModelDeclPtr, std::shared_ptr<Object>> m_singletons;

            std::shared_ptr<ErrorReporter> m_error_reporter;
            DocPtr m_current_document;

            bool m_abort;
    };
}
