
#pragma once

#include <memory>
#include <unordered_map>
#include <string>
#include <vector>
#include <openplx/Node.h>

namespace openplx
{
    class Document : public Node, public std::enable_shared_from_this<Document>
    {
        public:
            /* Factories */
            static std::shared_ptr<Document> create(const std::string& source_id, const std::vector<NodePtr>& members);

            /* Overrides */
            DocPtr asDocument() override;
            void accept(NodeVisitor& visitor) override;
            void unbind() override;

            /* Namespace */
            void appendToNamespace(std::string ns_str);
            const std::vector<std::string>& getNamespace() const;
            void copyNamespace(Document& other);
            std::string joinNamespace(const std::string& separator) const;
            std::string joinNamespaceWithPrefix(const std::string& separator, const std::string& prefix = "openplx") const;
            std::string joinNamespaceSkipFirst(const std::string& separator) const;
            bool compareNamespace(const Document& other) const;
            void cacheNsLookup(const std::string& key, NodePtr node, const DocPtr& source_document);
            NodePtr nsLookup(const std::string& key);
            NodePtr nsLookupConstOrVar(const std::string& key);
            void copyNsCacheFrom(Document& other);
            void clearNsCache();

            /* Members */
            void appendToMembers(NodePtr member);
            const std::vector<NodePtr>& getMembers() const;
            void removeMember(const NodePtr& member);
            void removeInvalidMembers();
            std::vector<ImportPtr> findImports() const;
            std::vector<NodePtr> findMembers(const std::string& name);
            NodePtr findFirstMemberOfType(const std::string& name, NodeType type) const;
            NodePtr findFirstMember(const std::string& name) const;
            NodePtr findLastMemberOfType(NodeType type) const;

            /* Source id */
            std::string getSourceId() const;
            void setSourceId(const std::string& source_id);

        private:
            Document();
            std::string m_source_id;
            std::vector<std::string> m_namespace;
            std::vector<NodePtr> m_members;
            std::unordered_map<std::string, NodePtr> m_ns_lookup;
    };
}
