#pragma once

#include <memory>
#include <optional>
#include <vector>
#include <unordered_map>
#include <openplx/AnalysisContext.h>
#include <openplx/BundleLookup.h>
#include <openplx/ErrorReporter.h>
#include <openplx/Type.h>
#include <openplx/Document.h>
#include <openplx/PrimitiveType.h>
#include <openplx/VarDeclaration.h>
#include <openplx/SymbolTree.h>

namespace openplx::Analysis
{
    class AnalysisContext {
        public:
            AnalysisContext(std::vector<BundleLookup> bundles, std::shared_ptr<ErrorReporter> error_reporter) {
                this->m_primitive_type_bool = PrimitiveType::create(PrimitiveTypeType::Bool);
                this->m_primitive_type_real = PrimitiveType::create(PrimitiveTypeType::Real);
                this->m_primitive_type_string = PrimitiveType::create(PrimitiveTypeType::String);
                this->m_primitive_type_int = PrimitiveType::create(PrimitiveTypeType::Int);
                this->m_bundles = bundles;
                this->m_error_reporter = error_reporter;
            }
            ~AnalysisContext();

            PrimitiveTypePtr primitive_type_real() {
                return this->m_primitive_type_real;
            }

            PrimitiveTypePtr primitive_type_string() {
                return this->m_primitive_type_string;
            }

            PrimitiveTypePtr primitive_type_int() {
                return this->m_primitive_type_int;
            }

            PrimitiveTypePtr primitive_type_bool() {
                return this->m_primitive_type_bool;
            }

            std::vector<DocPtr> findAllDocuments();
            std::vector<BundleLookup> getBundles();

            void cacheNsLookups();
            void clearNsLookups();
            std::vector<std::string> resolveNsPaths(Document& target, Document& source);
            void initSymbolTree(const std::vector<NodePtr>& root_nodes);
            SymTreePtr getSymbolTree() const;

            /**
             * Finds the modeldeclaration corresponding to fullpath in a document
             * (Because the same symbol_ref in two documents may point to different ModelDeclarations)
             *
             * @param document the document the symbol_ref was found in
             */
            ModelDeclPtr findModelDeclaration(Document& document, const std::vector<Token>& fullname);

            /**
             * Finds the type corresponding to fullname in a document
             * (The same name in two documents may point to different types)
             *
             * @param document the document the symbol_ref was found in
             */
            TypePtr findType(Document& document, const std::vector<Token>& fullname);

            ModelDeclPtr findModelDeclarationByNameIn(const std::string& name, Document& document);
            std::vector<NodePtr> findMembersInAllDocuments(const Token& name_token, Document& document);
            std::vector<VarAssignPtr> findTypeOf(const std::vector<Token>& segments, ModelDeclaration& model);
            static Token findFirstMissingType(const std::vector<Token> &segments, ModelDeclaration &model);
            OpOverloadPtr findBinaryOperator(TokenType op_type, const TypePtr& lhs, const TypePtr& rhs);
            OpOverloadPtr findUnaryOperator(TokenType op_type, const TypePtr& operand);

            /**
             * Finds all the model declarations (including itself) and var declarations that
             * is to be sorted for a specific model declarations.
             *
             * Note: Current implementation only returns the model declaration itself and all
             *       const model declarations. A better implementation would return
             *       root level var declarations as well, and filter out models and variables
             *       that are not referenced.
             *
             *  @param model_decl The model declaration which we want to find all
             *                    model and variable declarations that can be referenced.
             */
            std::vector<NodePtr> findTopoNodesFor(const ModelDeclPtr& model_decl);

            std::shared_ptr<ErrorReporter> getErrorReporter() {
                return m_error_reporter;
            }

            void invalidateDuplicateSymbols();
            void reportDuplicate(const std::string& key, const NodePtr& document_member, bool remove_from_document);

            /**
             * Decouples the nodes from the context, preventing the analysis context destructor
             * to deallocate them. The user is from this point responsible for calling `unbind`
             * on all documents to prevent memory leaks.
             */
            void release();

        private:
            void findTypeOfWithOffset(const std::vector<Token>& segments, ModelDeclaration& model, std::vector<VarAssignPtr>& result, size_t offset);

            PrimitiveTypePtr m_primitive_type_real;
            PrimitiveTypePtr m_primitive_type_string;
            PrimitiveTypePtr m_primitive_type_bool;
            PrimitiveTypePtr m_primitive_type_int;
            std::vector<BundleLookup> m_bundles;
            std::shared_ptr<ErrorReporter> m_error_reporter;
            SymTreePtr m_symbol_tree;
    };
}
