/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/TriaxialSignalSystemNode.h>

#include <random>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( ITriaxialSignalNoiseNode );
  AGX_DECLARE_POINTER_TYPES( TriaxialSignalNoiseNode );
  AGX_DECLARE_VECTOR_TYPES( TriaxialSignalNoiseNode );

  /**
  Interface for implementation details for triaxial signal system nodes carrying a random noise
  generator.
  */
  class AGXSENSOR_EXPORT ITriaxialSignalNoiseNode : public ITriaxialSignalSystemNode
  {
    public:
      /**
      Construct a trixaial signal noise node implementation.
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      ITriaxialSignalNoiseNode( agx::UInt32 seed = 67089u );

      /**
      Set the noise seed.
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      void setSeed( agx::UInt32 seed );

      /**
      \return seed used to configure the pseudo-random perturbation generation
      */
      agx::UInt32 getSeed() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual TriaxialSignalSystemNodeRef createTriaxialNodeProxy() override;
      virtual void synchronize( SystemNodeProxy& proxy, const CallbackData& data ) override;
      virtual void result( SystemNodeProxy& proxy, const CallbackData& data ) override;

      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSensor::ITriaxialSignalNoiseNode );
      virtual void store( agxStream::OutputArchive& out ) const override;
      virtual void restore( agxStream::InputArchive& in ) override;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::UInt32 m_seed;
      bool m_seedChanged;
  };

  /**
  A system node in the sensor processing tree carrying a triaxial signal, a random noise generator
  and an implementation for how to apply the noise.
  */
  class AGXSENSOR_EXPORT TriaxialSignalNoiseNode : public TriaxialSignalSystemNode
  {
    public:
      typedef std::mt19937_64 RandomGenerator;
      using result_type = RandomGenerator::result_type;

    public:
      /**
      \return minimum noise value that can be generated
      */
      inline static constexpr RandomGenerator::result_type min()
      {
        return RandomGenerator::min();
      }

      /**
      \return maximum noise value that can be generated
      */
      inline static constexpr RandomGenerator::result_type max()
      {
        return RandomGenerator::max();
      }

    public:
      /**
      Construct a triaxial signal noise node instance with the given \p implementation.
      \param implementation - behavioral implementation for this proxy (should not be nullptr)
      */
      TriaxialSignalNoiseNode( ITriaxialSignalNoiseNode* implementation );

      /**
      Set the noise seed.
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      void setSeed( agx::UInt32 seed );

      /**
      \return seed used to configure the pseudo-random perturbation generation
      */
      agx::UInt32 getSeed() const;

      /**
      \return generated noise value
      */
      RandomGenerator::result_type operator()();

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      TriaxialSignalNoiseNode();

      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::TriaxialSignalNoiseNode );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::UInt32 m_seed;
      agx::UInt64 m_generatedCount;
      RandomGenerator m_generator;
  };
}
