/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/SensorOutputHandler.h>
#include <agxSensor/TriaxialOutput.h>
#include <agxSensor/TriaxialSignalSystemNode.h>

#include <optional>
#include <utility>

namespace agxSensor
{
  using OptionalUniqueId = std::optional<size_t>;
  using OptionalTriaxialOutputMapping = std::optional<std::pair<size_t, TriaxialOutput*>>;

  AGX_DECLARE_POINTER_TYPES( TriaxialOutputHandler );

  /**
  Common base type for triaxial sensor output handlers.
  */
  class AGXSENSOR_EXPORT TriaxialOutputHandler : public SystemNode, public ISensorOutputHandler
  {
    public:
      /**
      Default constructor.
      */
      TriaxialOutputHandler();

      /**
      Add the given triaxial output to the output handler mapped to the given unique id.
      \param uniqueId - unique id > 0, id 0 is reserved and invalid to use
      \param output - triaxial output instance to add
      \return true if \p uniqueId > 0, \p uniqueId is unique and \p output != nullptr
      */
      bool add( size_t uniqueId, TriaxialOutput* output );

      /**
      Add the given triaxial output to the output handler. The added TriaxialOutput can later be
      accessed from get() by using the returned uniqueId.
      \param output - triaxial output instance to add
      \return the unique id assigned to the output on success, or nullopt on failure
      */
      OptionalUniqueId add( TriaxialOutput* output );

      /**
      Build matching triaxial output to a given data type T and add it mapped to the given unique
      id. This method verifies the size of the sum of data fields matches the size of T.
      \param uniqueId - unique id > 0, id 0 is reserved and invalid to use
      \return true if \p uniqueId > 0 and \p uniqueId is unique
      */
      template<typename T, TriaxialOutput::Field... field>
      bool add( size_t uniqueId );

      /**
      Build matching triaxial output to a given data type T. This method verifies the size of the
      sum of data fields matches the size of T. The created TriaxialOutput returned from this method
      can later be accessed from get() by using the returned uniqueId.
      \return a pair containing the unique id assigned to the output and the built triaxial output
      */
      template<typename T, TriaxialOutput::Field... field>
      OptionalTriaxialOutputMapping add();

      /**
      Access triaxial output instance given the unique id from when it was added.
      \param uniqueId - unique id for the output instance
      \return output instance for the given unique id if it exists, otherwise nullptr
      */
      TriaxialOutput* get( size_t uniqueId );

      /**
      Access triaxial output instance given the unique id from when it was added.
      \param uniqueId - unique id for the output instance
      \return output instance for the given unique id if it exists, otherwise nullptr
      */
      ISensorOutput* getBase( size_t uniqueId ) override;

      /**
      \return all added and active output modifiers
      */
      ITriaxialSignalSystemNodeRefVector getOutputModifiers() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSensor::TriaxialOutputHandler );

      virtual void store( agxStream::OutputArchive& out ) const override;
      virtual void restore( agxStream::InputArchive& in ) override;

    protected:
      using OutputTable = agx::HashTable<size_t, TriaxialOutputRef>;

    protected:
      bool addModifierInternal( ITriaxialSignalSystemNode* node );
      bool removeModifierInternal( ITriaxialSignalSystemNode* node );
      bool containsModifierInternal( const ITriaxialSignalSystemNode* node ) const;

      DOXYGEN_END_INTERNAL_BLOCK()

    protected:
      OutputTable m_outputs;
      TriaxialSignalSystemNodeRefVector m_outputModifiers;
      SystemNodeRef m_tailNode;
  };

  template<typename T, TriaxialOutput::Field... field>
  bool TriaxialOutputHandler::add( size_t uniqueId )
  {
    TriaxialOutputRef output = new TriaxialOutput();
    output->build<field...>();
    return add( uniqueId, output );
  }

  template<typename T, TriaxialOutput::Field... field>
  OptionalTriaxialOutputMapping TriaxialOutputHandler::add()
  {
    TriaxialOutputRef output = new TriaxialOutput();
    output->build<field...>();

    auto uniqueId = add( output );
    if ( uniqueId )
      return std::make_pair( *uniqueId, output );
    else
      return std::nullopt;
  }
}
